(ns hypercrud.service.core
  (:require [clojure.core.match :refer [match]]
            [datomic.api :as d]
            [hypercrud.service.pedestal-util :as pedestal-util]
            [hypercrud.service.util :as util]
            [io.pedestal.http.body-params :as body-params]
            [ring.util.response :as ring-resp]))


(defn enter [conn security-predicate tx user-token queries]
  (let [user user-token
        tx (if tx (Long/parseLong tx) (util/latest-tx conn))
        db (d/filter (d/as-of (d/db conn) tx) (partial security-predicate user))]
    (->> queries
         (map (fn [[query params pull-exp :as hc-query]]
                (let [pulled-tree (->> (apply d/q query db params)
                                       (d/pull-many db pull-exp))]
                  [hc-query pulled-tree])))
         (into {})
         util/wrap-hypercrud
         ring-resp/response)))


(defn transact! [conn validate-tx user-token user-tx]
  (let [user user-token
        db (d/db conn)                                      ; tx validation needs schema, so gets unfiltered db
        valid? (validate-tx db user user-tx)
        user-tx (map #(util/marshal-statement-types db %) user-tx)]
    (if-not valid?
      {:status 500 :headers {} :body "user tx failed validation"}

      (let [hc-tx [{:db/id (d/tempid :db.part/tx) :hypercrud/created-by user}]
            effect (d/transact conn (concat user-tx hc-tx))
            tx (-> @effect :db-after d/basis-t d/t->tx)]
        (ring-resp/response (util/wrap-hypercrud {:tx tx}))))))


(defn latest [conn]
  (ring-resp/response (util/latest-tx conn)))


(defn root-enter [transactor-uri root-sec
                  {{:keys [tx user-token]} :query-params
                   queries :body-params}]
  (enter (util/get-root-conn transactor-uri) root-sec tx user-token queries))


(defn root-transact! [transactor-uri validate-tx
                      {{:keys [user-token]} :query-params
                       user-tx :body-params}]
  (transact! (util/get-root-conn transactor-uri) validate-tx user-token user-tx))


(defn root-latest [transactor-uri]
  (latest (util/get-root-conn transactor-uri)))


(defn root-route [transactor-uri validate-tx root-sec]
  `["/api" {}
    ^:interceptors [~(body-params/body-params)
                    pedestal-util/combine-body-params
                    pedestal-util/auto-content-type]

    ["/enter" {:post [:enter (partial root-enter ~transactor-uri ~root-sec)]}]
    ["/transact" {:post [:transact (partial root-transact! ~transactor-uri ~validate-tx)]}]
    ["/latest" {:get [:latest #(root-latest ~transactor-uri)]}]])


(defn app-enter [transactor-uri root-sec
                 {{:keys [app-owner app-name]} :path-params
                  {:keys [tx user-token root-tx root-user-token]} :query-params
                  queries :body-params}]

  (let [root-conn (util/get-root-conn transactor-uri)
        root-tx (if root-tx (Long/parseLong root-tx) (util/latest-tx root-conn))
        root-db (d/filter (d/as-of (d/db root-conn) root-tx) (partial root-sec root-user-token))
        app-user (d/q '[:find [?app ...]
                        :in $ [?app-name ?app-owner]
                        :where [?app :app/name ?app-name]
                        [?app :hypercrud/created-by ?app-owner]]
                      root-db [app-name app-owner])
        ;{:keys [:app/security-predicate]} (d/touch app-user)
        app-security-predicate (constantly true)
        app-conn (util/get-app-conn transactor-uri app-owner app-name)]
    (enter app-conn app-security-predicate tx user-token queries)))


(defn app-transact! [transactor-uri root-sec
                     {{:keys [app-owner app-name]} :path-params
                      {:keys [user-token root-user-token]} :query-params
                      user-tx :body-params}]
  (let [root-conn (util/get-root-conn transactor-uri)
        root-db (d/filter (d/db root-conn) (partial root-sec root-user-token))
        app-user (d/q '[:find [?app ...]
                        :in $ [?app-name ?app-owner]
                        :where
                        [?app :app/name ?app-name]
                        [?app :hypercrud/created-by ?app-owner]]
                      root-db [app-name app-owner])
        ;{:keys [:app/validate-tx]} (d/touch app-user)
        app-validate-tx (constantly true)
        app-conn (util/get-app-conn transactor-uri app-owner app-name)]
    (transact! app-conn app-validate-tx user-token user-tx)))


(defn app-latest [transactor-uri {{:keys [app-owner app-name]} :path-params}]
  (latest (util/get-app-conn transactor-uri app-owner app-name)))


(defn app-route [transactor-uri root-sec]
  `["/api" {}
    ^:interceptors [~(body-params/body-params)
                    pedestal-util/combine-body-params
                    pedestal-util/auto-content-type]

    ["/enter" {:post [:app-enter (partial app-enter ~transactor-uri ~root-sec)]}]
    ["/transact" {:post [:app-transact (partial app-transact! ~transactor-uri ~root-sec)]}]
    ["/latest" {:get [:app-latest (partial app-latest ~transactor-uri)]}]])
