(ns hypercrud.ssr.render
  (:require [cljs.nodejs :as node]
            [promesa.core :as p]
            [clojure.set :as set]
            [clojure.string :as string]
            [reagent.core :as reagent]
            [hypercrud.client.core :refer [HypercrudClient]]
            [hypercrud.client.util :as util]))

(def cheerio (node/require "cheerio"))


(defn render-app-html [app-component indexed-schema service-root hc-client-state page-rel-path]
  (p/promise
    (fn [resolve reject]
      (let [user-hc-dependencies (atom #{})
            is-completed? #(let [all-dependencies-loaded? (->> @hc-client-state
                                                               ((juxt (comp keys :server-datoms)
                                                                      (comp keys :query-results)
                                                                      (comp keys :rejected)))
                                                               (reduce into #{})
                                                               (set/subset? @user-hc-dependencies))
                                 ; todo cannot use cmp-deps it is shared between requests
                                 rendering-is-settled? (empty? (-> @hc-client-state :cmp-deps))]
                            (and all-dependencies-loaded? rendering-is-settled?))
            client (HypercrudClient. service-root
                                     indexed-schema
                                     hc-client-state
                                     user-hc-dependencies
                                     (fn [_ comp]
                                       (try
                                         (reagent/render-to-string comp)
                                         (if (is-completed?)
                                           (do (println (str "html-ready"))
                                               (let [cache-only-client (HypercrudClient. service-root
                                                                                         indexed-schema
                                                                                         hc-client-state
                                                                                         user-hc-dependencies
                                                                                         (fn [cmp comp] (assert false "bug: force-update after html ready"))
                                                                                         [])
                                                     html (try
                                                            (reagent/render-to-string (app-component cache-only-client page-rel-path))
                                                            (catch js/Error e (reject e)))]
                                                 (resolve html))))
                                         (catch js/Error e (reject e))))
                                     [])]
        (let [html (try
                     (reagent/render-to-string (app-component client page-rel-path))
                     (catch js/Error e (reject e)))]
          (if (is-completed?)
            (resolve html)))))))


(defn evaluated-template [template app-html service-root-browser app-root state-str]
  (let [$ (.load cheerio template)]
    (-> ($ "#app-css") (.attr "href" (str app-root "static/styles.css")))
    (-> ($ "#root") (.html app-html))
    (-> ($ "#app-preamble") (.attr "src" (str app-root "static/preamble.js")))
    (-> ($ "#service-root") (.text service-root-browser))
    (-> ($ "#app-root") (.text app-root))
    (-> ($ "#initial-state") (.text state-str))
    (-> ($ "#app-js") (.attr "src" (str app-root "static/main.js")))
    (.html $)))


(defn render-html-async [app-component indexed-schema hc-client-state
                         root-rel-path service-root-browser service-root-node app-root template]
  (->
    (render-app-html app-component
                     indexed-schema
                     service-root-node
                     hc-client-state
                     (do (assert (string/starts-with? root-rel-path app-root))
                         (subs root-rel-path (count app-root))))
    (p/then
      (fn [app-html]
        (let [state @hc-client-state]
          (swap! hc-client-state update-in [:rejected] (constantly {})) ;clear rejections for refresh; this breaks interleaved requests since cache is shared
          (evaluated-template
            template
            app-html
            service-root-browser
            app-root
            (util/transit-encode state)))))

    (p/catch
      (fn [error]
        (let [errors (if (map? error) error {:render-to-string error})
              error-html (map (fn [[k v]]
                                (str "<li>"
                                     "<h4><code>" k "</code></h4>"
                                     "<pre>" (.-stack v) "</pre>"
                                     "</li>"))
                              errors)]
          (p/rejected
            (str
              "<html><body><h2>" (count errors) " Errors</h2>"
              "<ul>" (string/join "" error-html) "</ul></body></html>")))))))
