(ns hypercrud.ssr.render
  (:require [cljs.nodejs :as node]
            [clojure.set :as set]
            [clojure.string :as string]
            [hypercrud.client.core :as hc]
            [hypercrud.client.reagent :as hcr]
            [hypercrud.client.util :as util]
            [promesa.core :as p]
            [reagent.core :as reagent]))

(def cheerio (node/require "cheerio"))


(defmethod hcr/force-update! :default [cmp comp-fn]
  (reagent/render-to-string (comp-fn)))


(defn render-app-html [app-component indexed-schema service-root hc-client-state page-rel-path]
  (p/promise
    (fn [resolve reject]
      (let [request-state (atom {})
            client (hc/HypercrudClient. service-root indexed-schema hc-client-state request-state [])
            is-completed? #(let [all-dependencies-loaded? (->> @hc-client-state
                                                               ((juxt (comp keys :server-datoms)
                                                                      (comp keys :query-results)
                                                                      (comp keys :rejected)))
                                                               (reduce into #{})
                                                               (set/subset? (:user-hc-deps @(hc/ssr-context client))))
                                 rendering-is-settled? (= 0 (get-in @(hc/ssr-context client) [:cmp-deps] 0))]
                            (and all-dependencies-loaded? rendering-is-settled?))
            render-fn #(reagent/render-to-string (app-component client page-rel-path))]

        (add-watch (hc/ssr-context client) :poll-html-ready
                   (fn [k r o n]
                     (if (not= (:cmp-deps o) (:cmp-deps n))
                       (if (is-completed?)
                         (do (println "html ready")
                             (try (resolve (render-fn))
                                  (catch js/Error e (reject e))))
                         #_"else wait for pending promises"))))

        (try (let [html (render-fn)]
               (if (is-completed?)
                 (resolve html)
                 #_"else wait for pending promises"))
             (catch js/Error e (reject e)))))))


(defn evaluated-template [template app-html service-root-browser app-root state-str]
  (let [$ (.load cheerio template)]
    (-> ($ "#app-css") (.attr "href" (str app-root "static/styles.css")))
    (-> ($ "#root") (.html app-html))
    (-> ($ "#app-preamble") (.attr "src" (str app-root "static/preamble.js")))
    (-> ($ "#service-root") (.text service-root-browser))
    (-> ($ "#app-root") (.text app-root))
    (-> ($ "#initial-state") (.text state-str))
    (-> ($ "#app-js") (.attr "src" (str app-root "static/main.js")))
    (.html $)))


(defn render-html-async [app-component indexed-schema hc-client-state
                         root-rel-path service-root-browser service-root-node app-root template]
  (->
    (render-app-html app-component
                     indexed-schema
                     service-root-node
                     hc-client-state
                     (do (assert (string/starts-with? root-rel-path app-root))
                         (subs root-rel-path (count app-root))))
    (p/then
      (fn [app-html]
        (let [client-initial-state (dissoc @hc-client-state :pending)]
          (swap! hc-client-state update-in [:rejected] (constantly {})) ;clear rejections for refresh; this breaks interleaved requests since cache is shared
          (evaluated-template
            template
            app-html
            service-root-browser
            app-root
            (util/transit-encode client-initial-state)))))

    (p/catch
      (fn [error]
        (let [errors (if (map? error) error {:render-to-string error})
              error-html (map (fn [[k v]]
                                (str "<li>"
                                     "<h4><code>" k "</code></h4>"
                                     "<pre>" (.-stack v) "</pre>"
                                     "</li>"))
                              errors)]
          (p/rejected
            (str
              "<html><body><h2>" (count errors) " Errors</h2>"
              "<ul>" (string/join "" error-html) "</ul></body></html>")))))))
