;; Copyright 2008-2014 Red Hat, Inc, and individual contributors.
;; 
;; This is free software; you can redistribute it and/or modify it
;; under the terms of the GNU Lesser General Public License as
;; published by the Free Software Foundation; either version 2.1 of
;; the License, or (at your option) any later version.
;; 
;; This software is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;; Lesser General Public License for more details.
;; 
;; You should have received a copy of the GNU Lesser General Public
;; License along with this software; if not, write to the Free
;; Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
;; 02110-1301 USA, or see the FSF site: http://www.fsf.org.

(ns immutant.web.session
  "Functions for using the cluster-wide servlet store for sessions."
  (:use ring.middleware.session.store
        [immutant.web.internal :only [current-servlet-request]])
  (:require [ring.middleware.session       :as ring-session]
            [immutant.web.session.internal :as int]
            [immutant.util                 :as u])
  (:import javax.servlet.http.HttpSession))

(defn using-servlet-session? [session]
  "Returns true if the given servlet session is being used to store the ring session."
  (int/using-servlet-session? session))

(defn ^HttpSession servlet-session
  "Returns the servlet session for the current request."
  []
  (and current-servlet-request
       (.getSession current-servlet-request)))

(deftype
    ^{:doc "A ring SessionStore implementation that uses the session provided by the servlet container."}
    ServletStore [] SessionStore
  (read-session [_ _]
    (let [session (servlet-session)]
      (if-let [data (and session 
                         (.getAttribute session int/session-key))]
        data
        {})))
  (write-session [_ _ data]
    ;; TODO: it may be useful to store data as entries directly in the
    ;; session. TorqueBox does this to share the session data with
    ;; java servlet components.
    (when-let [session (servlet-session)]
      (.setAttribute session int/session-key data)
      (.getId session)))
  (delete-session [_ _]
    (when-let [session (servlet-session)]
      (.removeAttribute session int/session-key)
      (.invalidate session))))

(defn servlet-store
  "Instantiates and returns a ServletStore"
  []
  (ServletStore.))

(defn set-session-timeout!
  "Sets the session timeout (in minutes), overriding the default of 30
  minutes. Applies to all of the web endpoints deployed within an
  application, since they all share the same session."
  [minutes]
  (when-let [ctx (int/web-context)]
    (.setSessionTimeout ctx minutes)))

(defn session-timeout
  "Returns the current session timeout in minutes."
  []
  (when-let [ctx (int/web-context)]
    (.getSessionTimeout ctx)))

(defn ^{:valid-options
        #{:cookie-name :domain :http-only :max-age :path :secure}}
  set-session-cookie-attributes!
  "Set session cookie attributes. Accepts the following kwargs, many
   of which are analagous to the ring session cookie attributes
   [default]:

   :cookie-name  The name of the cookie [JSESSIONID]
   :domain       The domain name where the cookie is valid [nil]
   :http-only    Should the cookie be used only for http? [false]
   :max-age      The amount of time the cookie should be retained by the
                 client, in seconds [-1, meaning no max-age.
                 Cookie will be deleted on browser close.]
   :path         The path where the cookie is valid [the context path]
   :secure       Should the cookie be used only for secure connections? [false]

   This function can be called multiple times, and will only alter the
   attributes passed to it.

   These attributes apply to all of the web endpoints deployed within
   application, since they all share the same session."
  [& {:as attrs}]
  (u/validate-options set-session-cookie-attributes! attrs)
  (when-let [ctx (int/web-context)]
    (let [cookie (.getSessionCookie ctx)
          set-param (fn [key f]
                      (when (contains? attrs key)
                        (f cookie (attrs key))))]
      (set-param :cookie-name #(.setName %1 %2))
      (if (contains? attrs :cookie-name)
        (reset! int/cookie-name (:cookie-name attrs)))
      (set-param :domain      #(.setDomain %1 %2))
      (set-param :http-only   #(.setHttpOnly %1 (boolean %2)))
      (set-param :max-age     #(.setMaxAge %1 %2))
      (set-param :path        #(.setPath %1 %2))
      (set-param :secure      #(.setSecure %1 (boolean %2))))))

(defn session-cookie-attributes
  "Returns a map of the current session cookie attributes."
  []
  (int/session-cookie-attributes))



