package org.logolith.kzgo;

import com.sun.jna.Pointer;
import com.sun.jna.Structure;
import java.util.Arrays;
import java.util.List;

/**
 * JNA Structure mapping for the C buffer struct.
 * struct buffer {
 *     unsigned char* data;
 *     int len;
 * };
 */
public class CBuffer extends Structure {
    public Pointer data; // Maps to unsigned char*
    public int len;      // Maps to int

    public CBuffer() { super(); }

    public CBuffer(Pointer data, int len) {
        super();
        this.data = data;
        this.len = len;
    }

    @Override
    protected List<String> getFieldOrder() {
        return Arrays.asList("data", "len");
    }

    /** Marker interface for passing/returning CBuffer struct by value. */
    public static class ByValue extends CBuffer implements Structure.ByValue {}

    /** Marker interface for passing/returning CBuffer struct by reference (pointer). */
    public static class ByReference extends CBuffer implements Structure.ByReference {}

    /** Helper method to read bytes from the buffer, handling null pointers. */
    public byte[] readBytes() {
        if (this.data == null || this.len <= 0) {
            return new byte[0];
        }
        return this.data.getByteArray(0, this.len);
    }

    /** Helper to create a ByValue copy for passing to free_buffer */
    public ByValue asByValue() {
        ByValue byValue = new ByValue();
        byValue.data = this.data;
        byValue.len = this.len;
        return byValue;
    }
} 