(ns mentat.clerk-utils.show
  "Show utilities for Clerk."
  (:require [applied-science.js-interop :as j]
            [clojure.walk :as walk]
            [nextjournal.clerk #?(:clj :as :cljs :as-alias) clerk])
  #?(:cljs
     (:require-macros mentat.clerk-utils.show)))

(defmacro show-sci
  "Returns a form that executes all `exprs` in Clerk's SCI environment and renders
  the final form. If the final form evaluates to a vector, the vector is
  interpreted as a Reagent component.

  Else, the form is presented with `[nextjournal.clerk.viewer/inspect
  form]`. (To present a vector, manually wrap the final form in
  `[nextjournal.clerk.viewer/inspect ,,,]`.)

  Works in both `clj` and `cljs` contexts; in `cljs` this is equivalent to
  `clojure.core/comment`."
  [& exprs]
  (when-not (:ns &env)
    `(clerk/with-viewer
       {:transform-fn clerk/mark-presented
        :render-fn
        '(fn [_#]
           (let [result# (do ~@exprs)]
             (nextjournal.clerk.viewer/html
              (if (vector? result#)
                result#
                [nextjournal.clerk.render/inspect result#]))))}
       {})))

;; ## show-cljs macro
;;
;; For using compiled ClojureScript in a notebook.

(defn- stable-hash-form
  "Replaces gensyms and regular expressions with stable symbols for consistent
  hashing."
  [form]
  (let [!counter (atom 0)
        !syms    (atom {})]
    (walk/postwalk
     (fn [x]
       (cond #?(:cljs (regexp? x)
                :clj  (instance? java.util.regex.Pattern x))
             (symbol (str "stable-regexp-" (swap! !counter inc)))
             (and (symbol? x)
                  (not (namespace x)))
             (or (@!syms x)
                 (let [y (symbol (str "stable-symbol-" (swap! !counter inc)))]
                   (swap! !syms assoc x y)
                   y))
             :else x)) form)))

(def ^:no-doc stable-hash
  (comp hash stable-hash-form))

#?(:clj
   (def ^{:no-doc true
          :doc "This viewer takes function name generated by the macro body
     of [[show-cljs]] and loads the results.

    If the cljs code is not ready, shows a loading marker under the class
     `show-cljs-loading`.

    Once the CLJS code is ready, the cljs form is presented as either

    - a reagent form, if it's a vector and `:inspect true` is not set in the metadata
    - a form passed to `sicmutils.clerk.render/inspect` otherwise."}
     loading-viewer
     {:transform-fn
      nextjournal.clerk/mark-presented
      :render-fn
      '(fn render-var [fn-name]
         ;; ensure that a reagent atom exists for this fn
         (applied-science.js-interop/update-in!
          js/window
          [:show-cljs fn-name]
          (fn [x] (or x (reagent.core/atom {:loading? true}))))
         (let [res @(applied-science.js-interop/get-in js/window [:show-cljs fn-name])]
           (if (:loading? res)
             [:div.show-cljs-loading
              {:style {:color "rgba(0,0,0,0.5)"}}
              "Loading..."]
             (let [result (try ((:f res))
                               (catch js/Error e
                                 (js/console.error e)
                                 [nextjournal.clerk.render/error-view e]))]

               (if (and (vector? result)
                        (not (:inspect (meta result))))
                 [:div.show-cljs-loaded result]
                 [nextjournal.clerk.render/inspect result])))))}))

(defmacro show-cljs
  "CLJC macro that allows you to define forms in ClojureScript and make them
  available to Clerk's browser in a single place.

  Returns a form that executes all `exprs` and renders the final form.

  If the final form evaluates to a vector, the vector is interpreted as a
  Reagent component. (To present a vector, prepend the form with `^:inspect`
  metadata.)

  Else, the form is presented with `[nextjournal.clerk.viewer/inspect form]`.

  ## How it Works

  NOTE that this macro only makes sense when used inside of a cljc file, not in
  clj or cljs independently!

  When called from ClojureScript, emits all top-level `defn` forms and returns a
  thunk that executes all non-`defn` forms when called. As a side effect, this
  thunk is stored under `show_cljs.<hash-derived-name>` in `js/window`.

  When called from Clojure, generates the same `<hash-derived-name>` from the
  code forms and calls [[loading-viewer]] with this name."
  ([] nil)
  ([& exprs]
   (let [[defns others]
         ((juxt filter remove)
          #(when (seq? %)
             (= 'defn (first %)))
          exprs)]
     (if (empty? others)
       ;; This first branch contains only `defn` forms. Insert them into the
       ;; cljs side and do nothing on the JVM.
       (when (:ns &env)
         `(do ~@defns))
       ;; This branch handles side-effecting forms, and/or a final form we'd
       ;; like to render using Reagent.
       (let [fn-name (str *ns* "-" (mentat.clerk-utils.show/stable-hash others))]
         (if (:ns &env)
           `(do ~@defns
                ;; On the Clojurescript side, splice in all `defn` forms, and
                ;; generate a thunk that will run all side effects and return
                ;; the final non-defn for Reagent.
                (let [f# (fn [] ~@others)]
                  ;; Attempt to store this function in the window.
                  (j/update-in!
                   ~'js/window [:show-cljs ~fn-name]
                   (fn [x#]
                     (cond
                       ;; If this is the first insert, then cljs loaded before
                       ;; Clerk's server. Insert the thunk wrapped in an atom.
                       (not x#) (reagent.core/atom {:f f#})

                       ;; This branch means the JVM got there first. Reset the
                       ;; atom vs setting it up.
                       (:loading? @x#) (doto x# (reset! {:f f#}))

                       ;; Otherwise this is a no-op reload (since otherwise the
                       ;; hash would have changed and we'd be hitting a
                       ;; different slot in the window.)
                       :else x#)))
                  f#))
           ;; On the Clojure side, activate `loading-viewer` above with the
           ;; hash-generated symbol.
           `(clerk/with-viewer loading-viewer
              ~fn-name)))))))
