(ns msync.properties.xml
  (:require [clojure.string :as string]
            [clojure.java.io :refer [reader]]
            [clojure.data.xml :as xml]
            [clojure.zip :as z]
            [clojure.data.zip :as zip]
            [clojure.data.zip.xml :as zip-xml]))

(defrecord Property [name value])
(defrecord PropertyGroup [children])

(defn key->path
  "Convert a possibly dotted key to a seq of keywords."
  [k]
  (map keyword (string/split k #"\.")))

(defn get-attribute [node attr-name]
  (-> node :attrs attr-name))

(defn child-by-name [parent-node child-name]
  (let [children          (:content parent-node)
        matching-children (filter
                            (fn [n] (= (name child-name) (get-attribute n :name)))
                            children)]
    (assert (< (count matching-children) 2))
    (first matching-children)))

(defn by-name-path [node path]
  (let [components (key->path path)
        root-name  (first components)
        node-name  (get-attribute node :name)]
    (if-not (= node-name (name root-name))
      nil
      (loop [node node components (rest components)]
        (if-let [top-level (child-by-name node (first components))]
          (if-not (empty? (rest components))
            (recur top-level (rest components))
            top-level)
          nil)))))

(defn get-all-by-tag
  [node tag-name]
  (filter #(= tag-name (:tag %)) (:content node)))

(defn get-properties [node]
  (let [property-nodes (get-all-by-tag node :property)]
    (reduce (fn [accum-m property-node]
              (let [name  (-> property-node :attrs :name)
                    value (->> property-node :content (apply str))]
                (assoc accum-m name value)))
            {} property-nodes)))

(defn get-property-groups [node]
  (get-all-by-tag node :property-group))

(defn load-xml-file
  [file]
  (-> file reader xml/parse))