(ns org.nfrac.comportex.demos.identity
  (:require [org.nfrac.comportex.core :as core]
            [org.nfrac.comportex.protocols :as p]
            [org.nfrac.comportex.encoders :as enc]
            [org.nfrac.comportex.util :as util]
            [clojure.string :as str]
            #?(:clj [clojure.core.async :refer [put! >! <! go]]
                    :cljs [cljs.core.async :refer [put! >! <!]]))
  #?(:cljs (:require-macros [cljs.core.async.macros :refer [go]])))

(def bit-width 600)
(def n-on-bits 30)
(def motor-bit-width 10)
(def motor-n-on-bits 5)

(def test-text
  "
Look at this apple.
This apple is round.
This apple is red.
It is round and red.

Look at that apple.
That apple is round.
That apple is green.
It is also round but it is green not red.

This is a apple.
This is a apple and it is red.
This is a red apple.
That is a apple and it is green.
That is a green apple.

This apple is good to eat.
That apple is also good to eat.

A green apple is good to eat.
A red apple is good to eat.
But a brown apple is bad.
A brown apple is not good to eat.

This apple is not brown.
This apple is not bad.
That apple is also not brown.
That apple is also not bad.

A apple is a fruit.
A orange is a fruit.
A banana is also a fruit.

A apple is round.
A orange is round.
But a banana is not round.

This is a fruit.
This fruit is good to eat.
")

(def test-text-2
  "
Look at this dog.
This dog has a tail.
This dog is small.
This dog is not big.
This dog is white.

Look at that dog.
That dog has a tail.
That dog is big.
That dog is not small.
That dog is brown.

This is a dog.
This is a small dog.
That is a dog.
That is a big dog.
That is a round dog.

This dog is good to play with.
A small dog is good to play with.

A dog is a animal.
A person is a animal.
A apple is not a animal.
A fruit is not a animal.

A dog has a tail.
A person has not a tail.

This is a animal.
This animal is small.
This animal has a tail.

This is a small animal.

Look at this person.
This person is Jane.
Jane is a person.
Jane is good to play with.
Jane is not a dog.
Jane is a big person.
")

(defn parse-sentences
  [text*]
  (let [text (str/lower-case (str/trim text*))]
    (->> (str/split text #"[^\w]*\.+[^\w]*")
         (mapv #(str/split % #"[^\w']+")))))

(def spec
  {:column-dimensions [1000]
   :depth 8
   :distal-perm-init 0.21
   :use-feedback? true
   })

(def higher-level-spec-diff
  {:column-dimensions [1000]
   :depth 8
   :ff-max-segments 5})

(defn initial-inval
  [sentences]
  {:sentences sentences
   :position [0 0] ;; [sentence word]
   :value (get-in sentences [0 0])
   :action {:next-word-saccade -1
            :next-sentence-saccade -1}
   })

(defn next-position
  [[senti wordi] action]
  (cond
    (pos? (:next-sentence-saccade action))
    [(inc senti) 0]
    (neg? (:next-sentence-saccade action))
    [0 0]
    (pos? (:next-word-saccade action))
    [senti (inc wordi)]
    (neg? (:next-word-saccade action))
    [senti 0]))

(defn apply-action
  [inval]
  (let [new-posn (next-position (:position inval) (:action inval))
        new-value (get-in (:sentences inval) new-posn)]
    (assoc inval
           :position new-posn
           :value new-value)))

(def word-sensor
  [:value
   (enc/unique-encoder [bit-width] n-on-bits)])

(def word-motor-sensor
  [[:action :next-word-saccade]
   (enc/category-encoder [motor-bit-width] [1 -1])])

(def sentence-motor-sensor
  [[:action :next-sentence-saccade]
   (enc/category-encoder [motor-bit-width] [1 -1])])

(defn two-region-model
  ([]
   (two-region-model spec))
  ([spec]
   (core/region-network {:rgn-0 [:input :word-motor]
                         :rgn-1 [:rgn-0 :sentence-motor]}
                        (constantly core/sensory-region)
                        {:rgn-0 spec
                         :rgn-1 (merge spec higher-level-spec-diff)}
                        {:input word-sensor}
                        {:word-motor word-motor-sensor
                         :sentence-motor sentence-motor-sensor}
                        )))

(defn htm-step-with-action-selection
  [world-c control-c]
  (comment
    ;; TODO: on next release of core.async, replace this go block with
    (let [inval (if-let [xf (poll! control-c)]
                  (xf inval)
                  inval)
          ]))
  (go
    (loop []
      (if-let [xf (<! control-c)]
        (let [inval (<! world-c)]
          (>! world-c (xf inval))
          (recur)))))
  (fn [htm inval]
    (let [;; do first part of step, but not depolarise yet (depends on action)
          htm-a (-> htm
                    (p/htm-sense inval :sensory)
                    (p/htm-activate)
                    (p/htm-learn))
          [senti wordi] (:position inval)
          ;; work out what the next action (saccade) should be
          sentences (:sentences inval)
          sentence (get sentences senti)
          word (get sentence wordi)
          end-of-sentence? (= wordi (dec (count sentence)))
          end-of-passage? (= senti (dec (count sentences)))
          r0-lyr (get-in htm-a [:regions :rgn-0 :layer-3])
          r1-lyr (get-in htm-a [:regions :rgn-1 :layer-3])
          r0-burst-frac (/ (count (p/bursting-columns r0-lyr))
                           (count (p/active-columns r0-lyr)))
          ;r1-burst-frac (/ (count (p/bursting-columns r1-lyr))
          ;                 (count (p/active-columns r1-lyr)))
          sent-burst? (cond-> (:sentence-bursting? (:action inval))
                        ;; ignore burst on first word of sentence
                        (pos? wordi) (or (>= r0-burst-frac 0.50)))
          action* (cond
                    ;; not yet at end of sentence
                    (not end-of-sentence?)
                    {:next-word-saccade 1}

                    ;; end of sentence.

                    ;; sentence not yet learned, repeat sentence
                    sent-burst?
                    {:sentence-bursting? false}

                    ;; go to next sentence (not yet at end of passage
                    ;; same letter-motor signal as when repeating a word
                    (not end-of-passage?)
                    {:next-sentence-saccade 1
                     :sentence-bursting? false}

                    ;; reached end of passage
                    :else
                    {:next-sentence-saccade -1
                     :sentence-bursting? false}
                    )
          ;; next-word-saccade represents starting a sentence (-1) or continuing (1)
          ;; that is all that rgn-0 knows.
          action (merge {:next-word-saccade -1
                         :next-sentence-saccade 0
                         :sentence-bursting? sent-burst?}
                        action*)
          inval-with-action (assoc inval :action action
                                   :prev-action (:action inval))]
      ;; calculate the next position
      (let [new-inval (apply-action inval-with-action)]
        (put! world-c new-inval))
      ;; depolarise (predict) based on action, and update :input-value
      (cond-> htm-a
        true
        (p/htm-sense inval-with-action :motor)
        true
        (p/htm-depolarise)
        ;; reset first region's sequence when going on to new sentence
        (and end-of-sentence? (not sent-burst?))
        (update-in [:regions :rgn-0] p/break :tm)
        ))))
