(ns org.ozias.cljlibs.logging.logging
  (:require [colorize.core :as col]
            [clojure.string :as str]
            [taoensso.timbre :as timbre]))

(def ^:private nc (atom false))

(defn- fmt-output-fn
  [{:keys [level throwable message timestamp hostname ns]}
   & [{:keys [nofonts?] :as appender-fmt-output-opts}]]
  (format "%s %s %-6s [%s] - %s%s"
          timestamp hostname (-> level name str/upper-case) ns (or message "")
          (or (timbre/stacktrace throwable "\n" (when nofonts? {})) "")))

(defn- vec->str [tagged]
  (let [tags (filter keyword? tagged)
        strings (filter string? tagged)]
    ((->> tags (map #(partial col/color %)) (reduce comp))
     (->> strings (interpose " ") (apply str)))))

(defn- wrap-color [sv]
  (if (vector? sv)
    (if @nc
      (first (filter string? sv))
      (vec->str sv))
    sv))

(defn- message->colored-message [message]
  (let [message (if (vector? (first message)) message [message])]
    (->> message
         (map wrap-color)
         (interpose " ")
         (apply str))))

(defn reportc
  ([throwable & message]
     (timbre/report throwable (message->colored-message message)))
  ([message]
     (timbre/report (message->colored-message message))))

(defn fatalc
  ([throwable & message]
     (timbre/fatal throwable (message->colored-message message)))
  ([message]
     (timbre/fatal (message->colored-message message))))

(defn errorc
  ([throwable & message]
     (timbre/error throwable (message->colored-message message)))
  ([message]
     (timbre/error (message->colored-message message))))

(defn warnc
  ([throwable & message]
     (timbre/warn throwable (message->colored-message message)))
  ([message]
     (timbre/warn (message->colored-message message))))

(defn infoc
  ([throwable & message]
     (timbre/info throwable (message->colored-message message)))
  ([message]
     (timbre/info (message->colored-message message))))

(defn debugc
  ([throwable & message]
     (timbre/debug throwable (message->colored-message message)))
  ([message]
     (timbre/debug (message->colored-message message))))

(defn tracec
  ([throwable & message]
     (timbre/trace throwable (message->colored-message message)))
  ([message]
     (timbre/trace (message->colored-message message))))

(defn configure-logging
  "Configure timbre logging framework.

  The argument is a map of the format:

  {:options
   {... 
    :logfile \"/path/to/log\"  ;fully qualified path to logfile 
    :verbosity 0-3           ;output level: 0 - warn, 1 - info, 2 - debug, 3 - trace
    :no-color true/false     ;color support disabled/enabled
    :stdout true/false       ;log to stdout (false is default)
    :formatter formatterfn   ;formatting function (set timbre for example)
    ...}}

  All the keys are optional.  The default configuration will log info and above messages 
  to stdout. Note that if no logfile is supplied stdout true regardless of the map
  value.

  Evaluates to the passed in options map.  Useful for chaining with parse-opts from 
  tools.cli."
  [{{:keys [logfile verbosity no-color stdout formatter] 
     :or {verbosity 1 no-color false stdout false formatter fmt-output-fn}} :options :as options}]
  (reset! nc no-color)
  (let [stdout (if (nil? logfile) true stdout)
        options (if (contains? options :options)
                  (update-in options [:options] assoc 
                             :verbosity verbosity
                             :no-color no-color
                             :stdout stdout
                             :formatter formatter)
                  options)]
    (condp = verbosity
      0 (timbre/set-level! :warn)
      1 (timbre/set-level! :info)
      2 (timbre/set-level! :debug)
      3 (timbre/set-level! :trace))
    (timbre/set-config! [:fmt-output-fn] formatter)
    (timbre/set-config! [:appenders :standard-out :enabled?] stdout)
    (timbre/set-config! [:appenders :spit :enabled?] false)
    (timbre/set-config! [:shared-appender-config :spit-filename] logfile)
    (timbre/set-config! [:appenders :my-spit]
                        {:doc "Spit appender with newline"
                         :min-level nil
                         :enabled? true
                         :async? false
                         :rate-limit nil
                         :fn (fn [{:keys [ap-config output]}]
                               (when-let [filename (:spit-filename ap-config)]
                                 (try (spit filename (str output "\n") :append true)
                                      (catch java.io.IOException _))))})
    options))
