# mvnmeta
Fetch maven artifact metadata into a Clojure map

## Status
[![Build Status](https://travis-ci.org/CraZySacX/mvnmeta.png)](https://travis-ci.org/CraZySacX/mvnmeta)
[![Dependencies Status](http://jarkeeper.com/CraZySacX/mvnmeta/status.png)](http://jarkeeper.com/CraZySacX/mvnmeta)

### artifact-metadata
Fetch artifact metadata as a Clojure map.

The lone argument is a map with the following keys:
* ```url (required)``` - The url of the repository to fetch the metadata from.
* ```artifact-id (required)``` - The artifact-id of the artifact to fetch.  Note, if no group-id is supplied the url will be formed by duplicating the artifact-id (e.g. http://clojars.org/repo/clj-http/clj-http/maven-metadata.xml).
* ```group-id (optional)``` - The group-id of the artifact to fetch.
* ```filename (optional)``` - The filename to fetch (defaults to maven-metadata.xml).  This key is useful if you with to fetch maven-metadata-local.xml or a repository specific metadata file.
* ```username (optional)``` - The username to use if authentication is required by the repository.
* ```password (optional)``` - The password to use if authentication is required by the repository.
* ```passphrase (optional)``` - The passphrase to use if authentication is required by the repository. If you supply password and passphrase, password takes precedence.
* ```type (optional)``` - A key representing the type of authentication to use.  ```:basic``` for basic authentication, ```:digest``` for digest authentication.  Note, if you supply a valid username/password combo and no type, basic will be used by default.

```clojure
(ns example.ns
  (:require [org.ozias.cljlibs.mvnmeta.mvnmeta :as mvnmeta :only (artifact-metadata)]
            (org.ozias.cljlibs.mvnmeta.mvnmeta.protocol [file]
                                                        [httpx])))
; Note: Make sure to require the protocols you wish to support.
; Currently, http:, https:, and file: are supported.

; http (no group-id) - 'Fetches http://clojars.org/repo/clj-http/clj-http/maven-metadata.xml'
(mvnmeta/artifact-metadata {:url "http://clojars.org/repo"
                            :artifact-id "clj-http"})
; http
(mvnmeta/artifact-metadata {:url "http://clojars.org/repo"
                            :group-id "org.clojure"
                            :artifact-id "clojure"})
; https
(mvnmeta/artifact-metadata {:url "https://clojars.org/repo"
                            :group-id "org.clojure"
                            :artifact-id "clojure"})
; file
(mvnmeta/artifact-metadata {:url "file:///home/user/.m2/repository"
                            :group-id "org.clojure"
                            :artifact-id "clojure"})
; file with non-default filename
(mvnmeta/artifact-metadata {:url "file:///home/user/.m2/repository"
                            :group-id "org.clojure"
                            :artifact-id "clojure"
                            :filename "maven-metadata-local.xml"})
; http with basic auth
(mvnmeta/artifact-metadata {:url "http://some.url/artifactory/repo"
                            :group-id "org.clojure"
                            :artifact-id "clojure"
                            :username "user"
                            :password "password"})
                            
(mvnmeta/artifact-metadata {:url "http://some.url/artifactory/repo"
                            :group-id "org.clojure"
                            :artifact-id "clojure"
                            :username "user"
                            :password "password"
                            :type :basic})
; https with digest auth
(mvnmeta/artifact-metadata {:url "http://some.url/artifactory/repo"
                            :group-id "org.clojure"
                            :artifact-id "clojure"
                            :username "user"
                            :password "encrypted password"
                            :type :digest})
```

#### Multiple Artifacts
If you wish to fetch multiple artifacts in one call, you can use the ```:artifacts``` key in the map argument.  The value of this key should be a vector of artifact maps as defined above.  Any key defined at the top level will be used to fill out the artifacts if they are missing that key.  If both the artifact and top-level contain the same key, the nested artifact key takes precedence.

```clojure
; Fetch the metadata for org.clojure/clojure, org.clojure/data.xml, and org.clojure/tools.cli from 
; http://repo1.maven.org/maven2
(mvnmeta/artifact-metadata {:url "http://repo1.maven.org/maven2"
                            :group-id "org.clojure"
                            :artifacts [{:artifact-id "clojure"}
                                        {:artifact-id "data.xml"}
                                        {:artifact-id "tools.cli"}]})

; Fetch the metadata for org.clojure/clojure and org.clojure/data.xml from http://repo1.maven.org/maven2
; and org.ozias.cljlibs/mvnmeta from http://clojars.org/repo
(mvnmeta/artifact-metadata {:url "http://repo1.maven.org/maven2"
                            :group-id "org.clojure"
                            :artifacts [{:artifact-id "clojure"}
                                        {:artifact-id "data.xml"}
                                        {:url "http://clojars.org/repo"
                                         :group-id "org.ozias.cljlibs"
                                         :artifact-id "mvnmeta"}]})
```

Please see the [test file](https://github.com/CraZySacX/mvnmeta/blob/master/test/org/ozias/cljlibs/mvnmeta/testmvnmeta.clj#L7) for more examples.
