# mvnmeta
Fetch maven artifact metadata into a Clojure map

## Status
[![Build Status](https://travis-ci.org/CraZySacX/mvnmeta.png)](https://travis-ci.org/CraZySacX/mvnmeta)

## Current Version
In project.clj add the following to your dependencies

```[org.ozias.cljlibs/mvnmeta "0.1.5"]```

Then in the file you wish to use the library
```clojure
(ns example.ns
  (:require [org.ozias.cljlibs.mvnmeta.mvnmeta :as mvnmeta :only (artifact-metadata)]
            [org.ozias.cljlibs.mvnmeta.mvnmeta.protocol.file]
            [org.ozias.cljlibs.mvnmeta.mvnmeta.protocol.httpx]))
            
; Note: Make sure to require the protocols you wish to support.
; Currently, http:, https:, and file: are supported.
```
### artifact-metadata
Fetch artifact metadata as a Clojure map.

The lone argument is a map with the following keys:
* ```url (required)``` - The url of the repository to fetch the metadata from.
* ```artifact-id (required)``` - The artifact-id of the artifact to fetch.  Note, if no group-id is supplied the url will be formed by duplicating the artifact-id (e.g. http://clojars.org/repo/clj-http/clj-http/maven-metadata.xml).
* ```group-id (optional)``` - The group-id of the artifact to fetch.
* ```filename (optional)``` - The filename to fetch (defaults to maven-metadata.xml).  This key is useful if you with to fetch maven-metadata-local.xml or a repository specific metadata file.
* ```username (optional)``` - The username to use if authentication is required by the repository.
* ```password (optional)``` - The password to use if authentication is required by the repository.
* ```passphrase (optional)``` - The passphrase to use if authentication is required by the repository. If you supply password and passphrase, password takes precedence.
* ```type (optional)``` - A key representing the type of authentication to use.  ```:basic``` for basic authentication, ```:digest``` for digest authentication.  Note, if you supply a valid username/password combo and no type, basic will be used by default.
* ```artifacts (optional)``` - A vector of artifact definitions.  See the [Multiple Artifacts](#multiple-artifacts) section for example usage. 

#### Result
A map of the following format:

```{:repo-url {:artifact-coord {:lastUpdated [] :latest [] :release [] :versions []}}}```

```clojure
{:http://clojars.org/repo
 {:org.ozias.cljlibs/mvnmeta
  {:lastUpdated ["20140129184552"]
   :latest []
   :release ["0.1.1"]
   :versions ["0.1.0" "c02890c-SNAPSHOT" "d7b8de8-SNAPSHOT" "0.1.1" "44324fa-SNAPSHOT"]}}
 :http://repo1.maven.org/maven2
 {:org.clojure/clojure
  {:lastUpdated ["20131123001708"]
   :latest ["1.6.0-alpha3"]
   :release ["1.6.0-alpha3"]
   :versions ["1.0.0" "1.1.0" "1.2.0" "1.2.1" "1.3.0-alpha5" "1.3.0-alpha6" "1.3.0-alpha7"
              "1.3.0-alpha8" "1.3.0-beta1" "1.3.0-beta2" "1.3.0-beta3" "1.3.0-RC0" "1.3.0"
              "1.4.0-alpha1" "1.4.0-alpha2" "1.4.0-alpha3" "1.4.0-alpha4" "1.4.0-alpha5"
              "1.4.0-beta1" "1.4.0-beta2" "1.4.0-beta3" "1.4.0-beta4" "1.4.0-beta5"
              "1.4.0-beta6" "1.4.0-beta7" "1.4.0" "1.5.0-alpha1" "1.5.0-alpha2"
              "1.5.0-alpha3" "1.5.0-alpha4" "1.5.0-alpha5" "1.5.0-alpha6" "1.5.0-alpha7"
              "1.5.0-beta1" "1.5.0-beta2" "1.5.0-beta7" "1.5.0-beta8" "1.5.0-beta9"
              "1.5.0-beta10" "1.5.0-beta11" "1.5.0-beta12" "1.5.0-beta13" "1.5.0-RC1"
              "1.5.0-RC2" "1.5.0-RC3" "1.5.0-RC4" "1.5.0-RC5" "1.5.0-RC6" "1.5.0-RC14"
              "1.5.0-RC15" "1.5.0-RC16" "1.5.0-RC17" "1.5.0" "1.5.1" "1.6.0-alpha1"
              "1.6.0-alpha2" "1.6.0-alpha3"]}}}
```

#### Single Artifact
```clojure
; http (no group-id) - 'Fetches http://clojars.org/repo/clj-http/clj-http/maven-metadata.xml'
(mvnmeta/artifact-metadata {:url "http://clojars.org/repo"
                            :artifact-id "clj-http"})
; http
(mvnmeta/artifact-metadata {:url "http://clojars.org/repo"
                            :group-id "org.clojure"
                            :artifact-id "clojure"})
; https
(mvnmeta/artifact-metadata {:url "https://clojars.org/repo"
                            :group-id "org.clojure"
                            :artifact-id "clojure"})
; file
(mvnmeta/artifact-metadata {:url "file:///home/user/.m2/repository"
                            :group-id "org.clojure"
                            :artifact-id "clojure"})
; file with non-default filename
(mvnmeta/artifact-metadata {:url "file:///home/user/.m2/repository"
                            :group-id "org.clojure"
                            :artifact-id "clojure"
                            :filename "maven-metadata-local.xml"})
; http with basic auth
(mvnmeta/artifact-metadata {:url "http://some.url/artifactory/repo"
                            :group-id "org.clojure"
                            :artifact-id "clojure"
                            :username "user"
                            :password "password"})
                            
(mvnmeta/artifact-metadata {:url "http://some.url/artifactory/repo"
                            :group-id "org.clojure"
                            :artifact-id "clojure"
                            :username "user"
                            :password "password"
                            :type :basic})
; https with digest auth
(mvnmeta/artifact-metadata {:url "http://some.url/artifactory/repo"
                            :group-id "org.clojure"
                            :artifact-id "clojure"
                            :username "user"
                            :password "encrypted password"
                            :type :digest})
```

#### Multiple Artifacts
If you wish to fetch multiple artifacts in one call, you can use the ```:artifacts``` key in the map argument.  The value of this key should be a vector of artifact maps as defined above.  Any key defined at the top level will be used to fill out the artifacts if they are missing that key.  If both the artifact and top-level contain the same key, the nested artifact key takes precedence.

```clojure
; Fetch the metadata for org.clojure/clojure, org.clojure/data.xml, and org.clojure/tools.cli from 
; http://repo1.maven.org/maven2
(mvnmeta/artifact-metadata {:url "http://repo1.maven.org/maven2"
                            :group-id "org.clojure"
                            :artifacts [{:artifact-id "clojure"}
                                        {:artifact-id "data.xml"}
                                        {:artifact-id "tools.cli"}]})

; Fetch the metadata for org.clojure/clojure and org.clojure/data.xml from http://repo1.maven.org/maven2
; and org.ozias.cljlibs/mvnmeta from http://clojars.org/repo
(mvnmeta/artifact-metadata {:url "http://repo1.maven.org/maven2"
                            :group-id "org.clojure"
                            :artifacts [{:artifact-id "clojure"}
                                        {:artifact-id "data.xml"}
                                        {:url "http://clojars.org/repo"
                                         :group-id "org.ozias.cljlibs"
                                         :artifact-id "mvnmeta"}]})
```

Please see the [test file](https://github.com/CraZySacX/mvnmeta/blob/master/test/org/ozias/cljlibs/mvnmeta/testmvnmeta.clj#L7) for more examples.
