(ns org.ozias.cljlibs.scm.git
  (:require [clojure.string :refer [split trim]]
            [me.raynes.conch :refer [with-programs]]
            [org.ozias.cljlibs.scm.core :refer :all]
            [org.ozias.cljlibs.utils.core :refer [successful?]]))

(def ^{:doc "A vector of all supported git commands"}
  git-cmdv
  ["add" "add--interactive" "am" "annotate" "apply" "archimport" "archive"
   "bisect" "bisect--helper" "blame" "branch" "bundle" "cat-file" "check-attr"
   "check-ignore" "check-mailmap" "check-ref-format" "checkout" "checkout-index"
   "cherry" "cherry-pick" "citool" "clean" "clone" "column" "commit"
   "commit-tree" "config" "count-objects" "credential" "credential-cache"
   "credential-cache--daemon" "credential-gnome-keyring" "credential-store"
   "cvsexportcommit" "cvsimport" "cvsserver" "daemon" "describe" "diff"
   "diff-files" "diff-index" "diff-tree" "difftool" "difftool--helper"
   "fast-export" "fast-import" "fetch" "fetch-pack" "filter-branch"
   "fmt-merge-msg" "for-each-ref" "format-patch" "fsck" "fsck-objects"
   "gc" "get-tar-commit-id" "grep" "gui" "gui--askpass" "hash-object" "help"
   "http-backend" "http-fetch" "http-push" "imap-send" "index-pack" "init"
   "init-db" "instaweb" "log" "ls-files" "ls-remote" "ls-tree" "mailinfo"
   "mailsplit" "merge" "merge-base" "merge-file" "merge-index" "merge-octopus"
   "merge-one-file" "merge-ours" "merge-recursive" "merge-resolve"
   "merge-subtree" "merge-tree" "mergetool" "mktag" "mktree" "mv" "name-rev"
   "notes" "p4" "pack-objects" "pack-redundant" "pack-refs" "patch-id" "prune"
   "prune-packed" "pull" "push" "quiltimport" "read-tree" "rebase"
   "receive-pack" "reflog" "relink" "remote" "remote-ext" "remote-fd"
   "remote-ftp" "remote-ftps" "remote-http" "remote-https" "remote-testsvn"
   "repack" "replace" "request-pull" "rerere" "reset" "rev-list" "rev-parse"
   "revert" "rm" "send-email" "send-pack" "sh-i18n--envsubst" "shell" "shortlog"
   "show" "show-branch" "show-index" "show-ref" "stage" "stash" "status"
   "stripspace" "submodule" "svn" "symbolic-ref" "tag" "unpack-file"
   "unpack-objects" "update-index" "update-ref" "update-server-info"
   "upload-archive" "upload-pack" "var" "verify-pack" "verify-tag" "web--browse"
   "whatchanged" "write-tree"])

(gen-scm-test-fn "git")

(defn git-help-parse
  "parse the git help output for a list of supported commands"
  []
  (if (git?)
    (->> (nth (->> (with-programs [git] (git "help" "-a" {:seq true}))
                   (partition-by #(= "" %))
                   (filter #(not= (list "") %))) 2)
         (map trim)
         (map #(split % #"  +"))
         (reduce into [])
         (sort))
    (list)))

(gen-base-scm-fn "git")
(defmacro gen-git-fns
  "Generate the git functions"
  []
  (gen-scm-fns "git" git-cmdv))
(gen-git-fns)

(defn get-url
  "Get the git url for the repository at
  the given directory"
  [dir]
  (trim ((git-ls-remote :dir dir) "--get-url")))

(defn- local-type?
  "Is the branch at the repository in dir
  of the given local type (heads or tags)"
  [type dir branch]
  (successful?
    ((git-show-ref :dir dir :verbose true) "--quiet" (str "--" type) branch)))

(defn- remote-type?
  "Is the branch at the repository in dir
  of the given remote type (heads or tags)"
  [type dir branch]
  (let [url (get-url dir)]
    (successful?
      ((git-ls-remote :dir dir :verbose true)
       "--exit-code" (str "-" type) url branch))))

(defn local-head?
  "Is the branch at the repository in dir
  a local head"
  [dir branch]
  (local-type? "heads" dir branch))

(defn local-tag?
  "Is the branch at the repository in dir
  a local tag"
  [dir branch]
  (local-type? "tags" dir branch))

(defn remote-head?
  "Is the branch at the repository in dir
  a remote head"
  [dir branch]
  (remote-type? "h" dir branch))

(defn remote-tag?
  "Is the branch at the repository in dir
  a local tag"
  [dir branch]
  (remote-type? "t" dir branch))

(defn head?
  "Is the branch at the repository in dir
  a head"
  [dir branch]
  (or (local-head? dir branch) (remote-head? dir branch)))

(defn tag?
  "Is the branch at the repository in dir
  a tag"
  [dir branch]
  (or (local-tag? dir branch) (remote-tag? dir branch)))

(defn local?
  "Is the branch at the repository in dir
  local"
  [dir branch]
  (or (local-head? dir branch) (local-tag? dir branch)))

(defn remote?
  "Is the branch at the repository in dir
  remote"
  [dir branch]
  (or (remote-head? dir branch) (remote-tag? dir branch)))

(defn current-branch
  "Get the current branch for the repository in dir"
  [dir]
  (trim ((git-rev-parse :dir dir) "--abbrev-ref" "HEAD")))

(defn diff-last
  "Show the difference between the last commit and HEAD"
  [dir]
  ((git-diff :dir dir) "HEAD^" "HEAD"))
