(ns org.ozias.cljlibs.semver.semver
  (:require [clojure.string :as str :only (split)]))

(def ^:private semver-re
  #"(\d+\.\d+\.\d+)(-([A-Za-z0-9-.]+))?(\+([A-Za-z0-9-.]+))?")

(defn- svvec->svmap
  "Convert a semver vector version to a map.
  \"1.0.0\" -> {:major 1 :minor 0 :patch 0}

  Evaluates to a vector
  [{:major 1 :minor 0 :patch 0} pre-release buildmeta]"
  [[mmp pr bm]]
  (if (seq mmp)
    [(zipmap [:major :minor :patch] (str/split mmp #"\.")) pr bm]
    []))

(defn semver?
  "Checks if the given string is a valid semantic version (See http://semver.org).

  Evaluates to a vector of three values if the semantic version is valid:

  1. A map of the major.minor.patch version {:major 0 minor 1 :patch 0}
  2. The pre-release information (i.e alpha)
  3. The build metadata (i.e. 20140203111111)

  Evaluates to nil otherwise"
  [version]
  (if-let [match (and (seq version)(re-matches semver-re version))]
    (->> match
         (partition 2)
         (map last)
         (into [])
         svvec->svmap)))

(defn svvec->str
  "Convert a semver vector to a string."
  [[{:keys [major minor patch]} pr bm]]
  (str major "." minor "." patch 
       (if (seq pr) (str "-" pr))
       (if (seq bm) (str "+" bm))))

(defn- strpart
  "Splits the string into a lazy sequence of substrings, alternating
  between substrings that match the patthern and the substrings
  between the matches.  The sequence always starts with the substring
  before the first match, or an empty string if the beginning of the
  string matches.

  For example: (partition #\"[a-z]+\" \"abc123def\")
  returns: (\"\" \"abc\" \"123\" \"def\")"
  [^java.util.regex.Pattern re ^String s]
  (let [m (re-matcher re s)]
    ((fn step [prevend]
       (lazy-seq
        (if (.find m)
          (cons (.subSequence s prevend (.start m))
                (cons (re-groups m)
                      (step (+ (.start m) (count (.group m))))))
          (when (< prevend (.length s))
            (list (.subSequence s prevend (.length s)))))))
     0)))

(defn- parse-int [s]
  (Integer/parseInt (re-find #"\A-?\d+" s)))

(defn- nil-fill [v l]
  (loop [vec v]
    (if (= (count vec) l)
      vec
      (recur (conj vec nil)))))

(defn- char->str [c]
  (if (instance? java.lang.Character c)
    (str c)
    c))

(defn- numeric? [s]
  (string? (re-matches #"\d+" (char->str s))))

(defn- alpha? [s]
  (string? (re-matches #"[A-Za-z]+" (char->str s))))

(defn- space? [s]
  (string? (re-matches #" " (char->str s))))

(defn- identifier? [s]
  (string? (re-matches #"[0-9A-Za-z-]+" s)))

(defn- compare-numeric [x y]
  (let [x (parse-int (char->str x))
        y (parse-int (char->str y))]
    (cond
     (> x y) 1
     (= x y) 0
     (< x y) -1)))

(def cnm (memoize compare-numeric))

(defn- compare-char [x y]
  (cond
   (and (space? x)(char? y)) -1
   (and (char? x)(space? y)) 1
   (and (numeric? x)(numeric? y)) (cnm x y)
   (and (alpha? x)(alpha? y)) (compare x y)
   :else (compare x y)))

(def ccm (memoize compare-char))

(defn- space-fill [s l]
  (loop [st s]
    (if (= (count st) l)
      st
      (recur (str st " ")))))

(defn- trailing-digits? [x]
  (re-matches #"([a-zA-Z]+)([0-9]+)" x))

(defn- compare-with-trailing [[ax dx] [ay dy]]
  (let [res (first (filter (complement zero?) (map ccm ax ay)))]
    (if (nil? res)
      (compare-numeric dx dy)
      res)))

(def catm (memoize compare-with-trailing))

(defn- compare-alphanumeric [x y]
  (let [xd (.toLowerCase x)
        yd (.toLowerCase y)
        xt (trailing-digits? xd)
        yt (trailing-digits? yd)
        m (max (count xd) (count yd))
        xd (space-fill xd m)
        yd (space-fill yd m)]
    (if (and (seq xt)(seq yt))
      (catm (rest xt) (rest yt))
      (let [res (first (filter (complement zero?) (map ccm xd yd)))]
        (if (nil? res) 0 res)))))

(def cam (memoize compare-alphanumeric))

(defn- identifier-compare [x y]
  (cond
   ; A SNAPSHOT qualifier always has higher precedence (they are usually newer)
   (and (= x "SNAPSHOT")((complement =) y "SNAPSHOT")) 1
   (and (= x "SNAPSHOT")(= y "SNAPSHOT")) 0
   (and (= y "SNAPSHOT")((complement =) x "SNAPSHOT")) -1
   ; A nil qualifier always has lower precedence
   ; i.e. A longer dotted qualifier with equal predecessors
   ; has greater precedence.
   (and (nil? x)(seq y)) -1
   (and (nil? x)(nil? y)) 0
   (and (seq x)(nil? y)) 1
   ; Two numeric identifiers are compared numerically
   (and (numeric? x)(numeric? y)) (cnm x y)
   ; Otherwise a numeric identifier has lower precedence
   ; than an alphanumeric identifier.
   (and (numeric? x)(identifier? y)) -1
   (and (identifier? x)(numeric? y)) 1
   ; Else compare the identifers alphanumerically
   :else (cam x y)))

(def icm (memoize identifier-compare))

(defn- identifier-precedence
  "Split the identifiers on '.', fill each vector with
  nil to the max of both, and the compare each identifier.
  The equal identifiers are filtered out and the first non-zero
  result is grabbed.

  If the result is nil the identifiers have equal precedence (0)."
  [x y]
  (let [xv (clojure.string/split x #"\.")
        xy (clojure.string/split y #"\.")
        m (max (count xv) (count xy))
        xv (nil-fill xv m)
        xy (nil-fill xy m)
        res (first (filter (complement zero?) (map icm xv xy)))]
    (if (nil? res) 0 res)))

(def ipm (memoize identifier-precedence))

(defn- nil-precedence
  "A nil qualifier has precedence over
  any non-empty qualifier (i.e. 1.0.0 > 1.0.0-alpha)"
  [x y]
  (cond
   (and (nil? x)(seq y))  1
   (and (nil? x)(nil? y)) 0
   (and (seq x)(nil? y)) -1
   :else (ipm x y)))

(defn- fmap [f m]
  (into {} (for [[k v] m] [k (f v)])))

(defn- compare-semantic-versions [x y]
  (let [xv (fmap parse-int (first x))
        yv (fmap parse-int (first y))]
    (if (= (:major xv) (:major yv))
      (if (= (:minor xv) (:minor yv))
        (if (= (:patch xv) (:patch yv))
          (nil-precedence (second x) (second y))
          (compare (:patch xv) (:patch yv)))
        (compare (:minor xv) (:minor yv)))
      (compare (:major xv) (:major yv)))))

(def csvm (memoize compare-semantic-versions))
  
; Compare major.minor.patch-<qual> versions.
; major minor and patch are compared numerically.
; qualifiers are compared more extensively if
; the comparison is tied after major.minor.patch.
; 
; 1.0.0 < 1.0.1 < 1.1.0 < 2.0.1
(defn- cmpvers [x y]
  (let [x (semver? x)
        y (semver? y)]
    (cond
     ; If x isn't a semver and y is, y has precedence
     (and (empty? x)(seq y))   -1
     ; If x and y aren't semver then we can't compare
     (and (empty? x)(empty? y)) 0
     ; If x is a sever and y isn't, x has precedence
     (and (seq x)(empty? y))    1
     :else (csvm x y))))

(def compare-versions (memoize cmpvers))
