(ns org.ozias.cljlibs.semver.semver
  (:require [clojure.string :as str]
            [org.ozias.cljlibs.utils.core :refer [fmapv]]))

(def ^{:private true :doc "Semantic Version regular expression"}
  semver-re #"(\d+\.\d+\.\d+)(-([A-Za-z0-9-.]+))?(\+([A-Za-z0-9-.]+))?")

(defn- svvec->svmap
  "Convert a semver vector version to a map.
  \"1.0.0\" -> {:major 1 :minor 0 :patch 0}

  Evaluates to a vector
  [{:major 1 :minor 0 :patch 0} pre-release buildmeta]"
  [[mmp pr bm]]
  (if (seq mmp)
    [(zipmap [:major :minor :patch] (str/split mmp #"\.")) pr bm]
    []))

(defn semver?
  "Checks if the given string is a valid semantic version
  (See http://semver.org).

  Evaluates to a vector of three values if the semantic version is valid:

  1. A map of the major.minor.patch version {:major 0 minor 1 :patch 0}
  2. The pre-release information (i.e alpha)
  3. The build metadata (i.e. 20140203111111)

  Evaluates to nil otherwise"
  [version]
  (if-let [match (and (seq version) (re-matches semver-re version))]
    (->> match
         (partition 2)
         (map last)
         (into [])
         svvec->svmap)))

(defn- svvec->str
  "Convert a semver vector to a string."
  [[{:keys [major minor patch]} pr bm]]
  (str major "." minor "." patch
       (if (seq pr) (str "-" pr))
       (if (seq bm) (str "+" bm))))

(defn- parse-int
  "parse an integer from a string"
  [s]
  (Integer/parseInt (re-find #"\A-?\d+" s)))

(defn- nil-fill
  "Fill a vector with nil to a given length"
  [v l]
  (loop [vec v]
    (if (= (count vec) l)
      vec
      (recur (conj vec nil)))))

(defn- char->str
  "Convert a character to a string"
  [c]
  (if (instance? java.lang.Character c)
    (str c)
    c))

(defn- numeric?
  "Is the given string numeric?"
  [s]
  (string? (re-matches #"\d+" (char->str s))))

(defn- alpha?
  "Is the given string alpha only?"
  [s]
  (string? (re-matches #"[A-Za-z]+" (char->str s))))

(defn- space?
  "Is the given string only a space?"
  [s]
  (string? (re-matches #" " (char->str s))))

(defn- identifier?
  "Is the given string a semver identifier?"
  [s]
  (string? (re-matches #"[0-9A-Za-z-]+" s)))

(defn- compare-numeric
  "Compare numeric strings"
  [x y]
  (let [x (parse-int (char->str x))
        y (parse-int (char->str y))]
    (cond
      (> x y) 1
      (= x y) 0
      (< x y) -1)))

(def ^{:private true :doc "Memoization of compare-numeric"}
  cnm (memoize compare-numeric))

(defn- compare-char
  "Compare to characters"
  [x y]
  (cond
    (and (space? x) (char? y)) -1
    (and (char? x) (space? y)) 1
    (and (numeric? x) (numeric? y)) (cnm x y)
    (and (alpha? x) (alpha? y)) (compare x y)
    :else (compare x y)))

(def ^{:private true :doc "Memoization of compare-char"}
  ccm (memoize compare-char))

(defn- space-fill
  "Fill a given string with spaces to the given length"
  [s l]
  (loop [st s]
    (if (= (count st) l)
      st
      (recur (str st " ")))))

(defn- trailing-digits?
  "Does the given string have trailing digits?"
  [x]
  (re-matches #"([a-zA-Z]+)([0-9]+)" x))

(defn- compare-with-trailing
  "Compare to strings with trailing digits."
  [[ax dx] [ay dy]]
  (let [res (first (remove zero? (map ccm ax ay)))]
    (if (nil? res)
      (compare-numeric dx dy)
      res)))

(def ^{:private true :doc "Memoization of compare-with-trailing"}
  catm (memoize compare-with-trailing))

(defn- compare-alphanumeric
  "Compare two alphanumeric strings per semver"
  [^String x ^String y]
  (let [xd (.toLowerCase x)
        yd (.toLowerCase y)
        xt (trailing-digits? xd)
        yt (trailing-digits? yd)
        m (max (count xd) (count yd))
        xd (space-fill xd m)
        yd (space-fill yd m)]
    (if (and (seq xt) (seq yt))
      (catm (rest xt) (rest yt))
      (let [res (first (remove zero? (map ccm xd yd)))]
        (if (nil? res) 0 res)))))

(def ^{:private true :doc "Memoization of compare-alphanumeric"}
  cam (memoize compare-alphanumeric))

(defn- identifier-compare
  "Compare to semver identifiers"
  [x y]
  (cond
    ; A SNAPSHOT qualifier always has higher precedence (they are usually newer)
    (and (= x "SNAPSHOT") (not= y "SNAPSHOT")) 1
    (and (= x "SNAPSHOT") (= y "SNAPSHOT")) 0
    (and (= y "SNAPSHOT") (not= x "SNAPSHOT")) -1
    ; A nil qualifier always has lower precedence
    ; i.e. A longer dotted qualifier with equal predecessors
    ; has greater precedence.
    (and (nil? x) (seq y)) -1
    (and (nil? x) (nil? y)) 0
    (and (seq x) (nil? y)) 1
    ; Two numeric identifiers are compared numerically
    (and (numeric? x) (numeric? y)) (cnm x y)
    ; Otherwise a numeric identifier has lower precedence
    ; than an alphanumeric identifier.
    (and (numeric? x) (identifier? y)) -1
    (and (identifier? x) (numeric? y)) 1
    ; Else compare the identifers alphanumerically
    :else (cam x y)))

(def ^{:private true :doc "Memoization of identifier-compare"}
  icm (memoize identifier-compare))

(defn- identifier-precedence
  "Split the identifiers on '.', fill each vector with
  nil to the max of both, and the compare each identifier.
  The equal identifiers are filtered out and the first non-zero
  result is grabbed.

  If the result is nil the identifiers have equal precedence (0)."
  [x y]
  (let [xv (clojure.string/split x #"\.")
        xy (clojure.string/split y #"\.")
        m (max (count xv) (count xy))
        xv (nil-fill xv m)
        xy (nil-fill xy m)
        res (first (remove zero? (map icm xv xy)))]
    (if (nil? res) 0 res)))

(def ^{:private true :doc "Memoization of identifier-precedence"}
  ipm (memoize identifier-precedence))

(defn- nil-precedence
  "A nil qualifier has precedence over
  any non-empty qualifier (i.e. 1.0.0 > 1.0.0-alpha)"
  [x y]
  (cond
    (and (nil? x) (seq y)) 1
    (and (nil? x) (nil? y)) 0
    (and (seq x) (nil? y)) -1
    :else (ipm x y)))

(defn- compare-semantic-versions
  "Compare two semantic versions"
  [x y]
  (let [xv (fmapv parse-int (first x))
        yv (fmapv parse-int (first y))]
    (if (= (:major xv) (:major yv))
      (if (= (:minor xv) (:minor yv))
        (if (= (:patch xv) (:patch yv))
          (nil-precedence (second x) (second y))
          (compare (:patch xv) (:patch yv)))
        (compare (:minor xv) (:minor yv)))
      (compare (:major xv) (:major yv)))))

(def ^{:private true :doc "Memoization of compare-semantic-versions"}
  csvm (memoize compare-semantic-versions))

(defn- cmpvers
  "Compare major.minor.patch-<qual> versions.
  major minor and patch are compared numerically.
  qualifiers are compared more extensively if
  the comparison is tied after major.minor.patch.

  1.0.0 < 1.0.1 < 1.1.0 < 2.0.0"
  [x y]
  (let [x (semver? x)
        y (semver? y)]
    (cond
      ; If x isn't a semver and y is, y has precedence
      (and (empty? x) (seq y)) -1
      ; If x and y aren't semver then we can't compare
      (and (empty? x) (empty? y)) 0
      ; If x is a sever and y isn't, x has precedence
      (and (seq x) (empty? y)) 1
      :else (csvm x y))))

(def ^{:doc "Compare major.minor.patch-<qual> versions.
  major minor and patch are compared numerically.
  qualifiers are compared more extensively if
  the comparison is tied after major.minor.patch.

  1.0.0 < 1.0.1 < 1.1.0 < 2.0.0"}
  compare-versions (memoize cmpvers))
