(ns org.ozias.cljlibs.utils.core
  (:require [clojure.string :refer (trim)]
            [environ.core :refer :all]
            [me.raynes.conch :refer (with-programs)]))

(defn os?
  "true if the os.name environment variable
  starts with the given name"
  [name]
  (.startsWith (env :os-name) name))

(defn linux?
  "true if the os.name environment variable
  starts with 'Linux'"
  []
  (os? "Linux"))

(defn mac?
  "true if the os.name environment variable
  starts with 'Mac'"
  []
  (os? "Mac"))

(defn windows?
  "true if the os.name environment variable
  starts with 'Windows'"
  []
  (os? "Windows"))

(defn- os-key-dispatcher
  "Used to generate a key based on os type to
  be used as dispatching function."
  []
  (cond
   (linux?) :linux
   (mac?) :mac
   (windows?) :windows))

(defn- nix-cpu-arch []
  (with-programs [uname]
    (let [arch (trim (uname "-m"))]
      (if (= "x86_64" arch)
        64
        (if (= "i686" arch)
          32)))))

(defmulti cpu-arch
  "evaluates to the number of bits for the current cpu architecture"
  os-key-dispatcher)
    
(defmethod cpu-arch :windows []
  (let [arch (env :processor-architecture)]
    (if (= "AMD64" arch)
      64
      (if (= "x86" arch)
        32))))

(defmethod cpu-arch :linux []
  (nix-cpu-arch))

(defmethod cpu-arch :max []
  (nix-cpu-arch))

(defn cpu-arch-64?
  "true if the cpu architecture is 64-bits"
  []
  (= 64 (cpu-arch)))

(defn cpu-arch-32?
  "true if the cpu architecture is 32-bits"
  []
  (= 32 (cpu-arch)))

(defn- nix-cores []
  (with-programs [nproc]
    (-> (nproc) trim Integer/parseInt)))

(defmulti cores
  "evalutes to the number of processor cores"
  os-key-dispatcher)

(defmethod cores :windows []
  (Integer/parseInt (env :number-of-processors)))

(defmethod cores :linux []
  (nix-cores))

(defmethod cores :mac []
  (nix-cores))

(defn assoc-if
  "Same as assoc, but skip the assoc if v is nil"
  [m & kvs]
  (->> kvs (partition 2) (filter second) (map vec) (into m)))

(defn successful?
  "checks a conch output map for an exit code of 0"
  [omap]
  (= 0 @(:exit-code omap)))

(defn prseq [s]
  (doseq [line s]
    (println line)))
