(ns leiningen.archaic.describe
  (:require [clojure.string :as str :only (join)]
            (leiningen.archaic.utils [io :refer :all]
                                     [repo :refer :all])
            [org.ozias.cljlibs.logging.logging :refer (infoc tracec warnc)]
            [org.ozias.cljlibs.semver.semver :as sv]))

(defn- filter-metadata [type]
  (fn [metadata]
    (cond
     (:release type) (:release metadata)
     (:snapshot type) (into [] (filter #(and (sv/semver? %)(snapshot? %)) (:versions metadata)))
     (:qualified type) (into [] (filter #(and (sv/semver? %)(qualified? %)) (:versions metadata)))
     (:any type) (into (:release metadata) (filter sv/semver? (:versions metadata))))))

(def ^:private filter-releases (filter-metadata {:release true}))
(def ^:private filter-snapshots (filter-metadata {:snapshot true}))
(def ^:private filter-qualified (filter-metadata {:qualified true}))
(def ^:private filter-any (filter-metadata {:any true}))

(defn- merge-sort-metas [metas]
  (->> metas
      (remove nil?)
      (reduce into)
      set
      (sort sv/compare-versions)
      reverse))

(defn- describe-artifact [project {:keys [options] :as opts}]
  (fn [artifact]
    (let [options (assoc options :any true)
          repos (resolve-repos project)
          metas (for [repo repos]
                  (fetch-artifact-meta repo (symbol artifact)))
          rel (merge-sort-metas (map filter-releases metas))
          snap (merge-sort-metas (map filter-snapshots metas))
          qual (merge-sort-metas (map filter-qualified metas))
          any (merge-sort-metas (map filter-any metas))]
      (warnc :green "Getting version information for "
             :green :bold "[" artifact "] "
             :green "from " (count repos) " repositories")
      (warnc :green "    * Version(s) found: " :yellow (count any))
      (warnc :green "    * Latest release:   " :yellow (first rel))
      (warnc :green "    * Latest SNAPSHOT:  " :yellow (first snap))
      (warnc :green "    * Latest qualified: " :yellow (first qual))
      (warnc :green "    * All versions:     "
             :magenta "[" (str/join ", " any) "]\n"))))

(defn run-task [project {:keys [arguments] :as opts}]
  (let [args (rest arguments)
        desc (describe-artifact project opts)]
    (if (empty? args)
      (-> "Please specify an artifact to describe"
          (arcabort opts))
      (dorun (map desc args)))))
  
