(ns leiningen.archaic
  (:require [clojure.string :as str]
            [clojure.tools.cli :as cli :refer (parse-opts)]
            (leiningen.archaic [check :as check]
                               [proj :as prj]
                               [profiles :as prf]
                               [describe :as desc]
                               [upgrade :as upgrade]
                               [usage :as usage]
                               [version :as ver])
            [leiningen.archaic.utils.io :refer (arcabort exists?)]            
            [org.ozias.cljlibs.logging.logging :refer (configure-logging reportc tracec)]
            [taoensso.timbre :as timbre :only (stacktrace)]))

(def ^:private logging-options
  {:stdout true 
   :formatter (fn [{:keys [throwable message]}]
                (format "%s%s" (or message "") (or (timbre/stacktrace throwable "\n") "")))})

(def ^:private cli-options
  [["-g" "--aggressive" "Check all available repositories" :default false]
   ["-f" "--first" "Stop checking after an artifact is found in any repository (NOT IMPLEMENTED)" :default false]
   ["-a" "--all" "Check all artifacts" :default false]
   ["-d" "--dependencies" "Check only top-level and profile dependency artifacts" :default false]
   ["-p" "--plugins" "Check only top-level and profile plugin artifacts" :default false]
   ["-C" "--no-clojure" "Exclude Clojure [org.clojure/clojure] in checks" :default false]
   ["-R" "--recursive" "Search recursively for 'project.clj' files" :default false]
   ["-r" "--release" "Check release (i.e. 1.0.0) versions" :default false]
   ["-q" "--qualified" "Check qualified (i.e. x.x.x-alpha) versions" :default false]
   ["-s" "--snapshot" "Check snapshot (i.e. x.x.x-SNAPSHOT) versions" :default false]
   ["-A" "--any" "Check SNAPSHOT, qualified, and release versions" :default false]
   ["-F" "--force" "Run the 'upgrade' task without interaction" :default false]
   ["-t" "--tests" "Run tests after upgrading (NOT IMPLEMENTED)" :default false]
   ["-P" "--print" "Print the results of the 'upgrade' tasks (no write)" :default false]
   ["-c" "--no-color" "Turn off terminal colors" :default false]
   ["-u" "--usage" "Show lein-archaic usage information" :default false]
   ["-v" nil "Output verbosity level; may be specified up to 3 times to increase output"
    :id :verbosity
    :default 0
    :assoc-fn (fn [m k _] (update-in m [k] inc))]
   [nil "--version" "Show lein-archaic version information" :default false]])

(def ^:private verdeferror
  "Only one of -r, -s, -q, or -A should be supplied.  If none are supplied, -r is assumed.")

(def ^:private artdeferror
  "Only one of -a, -d, or -p should be specified.  If none are supplied, -a is assumed.")

(def ^:private repodeferror
  "Only one of -g, or -f should be specified.  If none are supplied, -g is assumed.")

(def ^:private tasks
  {:check    check/run-task 
   :project  prj/run-task 
   :profiles prf/run-task
   :describe desc/run-task
   :upgrade  upgrade/run-task
   :usage    usage/run-task
   :version  (fn [_ _] (reportc :green (str "lein-archaic (git version: " ver/version ")")))})

(defn- args->kw 
  "If the first argument is a path, run check.
  Else, if the first argument is a string, convert it to a keyword
  Else, run check"
  [arguments]
  (let [f (first arguments)]
    (if (and f (exists? f))
      :check
      (if f
        (keyword f)           
        :check))))

(defn- only-one [arglst error opts]
  (let [tlst (filter true? arglst)]
    (if (= (count tlst) 1)
      opts
      (->> error
           (conj (:error opts))
           (assoc-in opts [:errors])))))

(defn- check-default [opts arglst defaultkey error]
  (if (every? false? arglst)
    (assoc-in opts [:options defaultkey] true)
    (only-one arglst error opts)))
  
(defn- check-defaults
  [{{:keys [aggressive first all dependencies plugins release snapshot qualified any]} :options :as opts}]
  (-> opts
      (check-default (list aggressive first) :aggressive repodeferror)
      (check-default (list all dependencies plugins) :all artdeferror)
      (check-default (list release snapshot qualified any) :release verdeferror)))

(defn- pprint-opts [options]
  (->> (for [[k v] options] (if (and v (not (fn? v))) (str (clojure.core/name k) " " v)))
       (remove nil?)
       (str/join ", ")))

(defn ^:higher-order ^:no-project-needed archaic
  "Check for archaic artifacts in your project or user profile. 
  
   Usage:

     lein archaic [<options>] [<task>]

   Options:
  
     Control how repositories are searched
       
       -g, --aggressive    Check all available repositories (default).
       -f, --first         Stop checking after an artifact is found in any repository. (NOT IMPLEMENTED)

     Control which artifacts are checked

       -a, --all          Check all artifacts (default).
       -d, --dependencies Check only top-level and profile dependency artifacts
                          (exclude plugins).
       -p, --plugins      Check only top-level and profile plugin artifacts 
                          (exclude dependencies).
       -C, --no-clojure   Exclude Clojure (org.clojure/clojure) in checks.
       -R, --recursive    Search for 'project.clj' files recursively.

     Control which artifact versions are checked from the repository

       -r, --release      Check release (i.e. 1.0.0) versions (default).
       -q, --qualified    Check qualified (i.e. '*-alpha*') and release versions.
       -s, --snapshots    Check SNAPSHOT (i.e. '*-SNAPSHOT') and release versions.
       -A, --any          Check SNAPSHOT, qualified, and release versions.
       
     Control the upgrade task

       -F, --force        Run the 'upgrade' task without interaction.
       -t, --tests        Run tests after upgrading. (NOT IMPLEMENTED)
       -P, --print        Print result of 'upgrade' task instead of writing.

     Control output
       -c, --no-color     Disable colorized output.
       -u, --usage        Show lein-archaic usage information.
           --version      Show lein-archaic version information.
  
   Tasks:

     check                Check for archaic artifacts in the current project and in 
                          ~/.lein/profiles.clj (default).
     project              Check for archaic artifacts in the current project.
     profiles             Check for archaic artifacts in ~/.lein/profiles.clj.
     describe <artifact>  List all known versions of a given artifact (i.e. org.clojure/clojure).
     upgrade              Replace the archaic artifacts in the current project (project.clj) and 
                          in ~/.lein/profiles.clj.
     upgrade project      Replace the archaic artifacts in the current project.
     upgrade profiles     Replace the archaic artifacts in ~/.lein/profiles.clj.
     usage                Show lein-archaic usage information.
     version              Show lein-archaic version information."
  [{:keys [name] :as project} & args]
  (let [{:keys [options errors arguments] :as opts}
        (-> args 
            (parse-opts cli-options)
            check-defaults
            (update-in [:options] merge logging-options) 
            configure-logging)]
    (cond
     (:version options) ((:version tasks) nil nil)
     (:usage options) ((:usage tasks) nil opts)
     errors ((:usage tasks) nil opts)
     :else (let [tkw (args->kw arguments)
                 taskname (clojure.core/name tkw)
                 task (tkw tasks)]             
             (if task
               (do
                 (tracec :green "Options:" :reset " " :yellow (pprint-opts options))
                 (tracec :green "Task:" :reset " " :yellow taskname "\n")
                 (task project opts))
               (-> (str "Unknown task: '" taskname "'!")
                   (arcabort opts)))))))
