(ns lein-git-version.plugin
  (:require [clojure.java.io :as io]
            [clojure.string :as str]
            [leiningen.core.main]
            [leiningen.git-version :refer [get-version get-file-version]]
            [robert.hooke :refer (add-hook)]))

(def ^:private default-keys {:assoc-in-keys [[:version]]
                             :filepath nil
                             :filename "version.clj"})
                   
(defn- version-file [{:keys [git-version group name source-paths]}]
  (let [group (str/replace group "-" "_")
        name (str/replace name "-" "_")
        config (merge default-keys git-version)
        srcpath (first (filter seq (map #(re-find #".*src$" %) source-paths)))
        filename (:filename config)
        filepath (:filepath config)
        pathvec (if (seq filepath)
                  (conj [] 
                        srcpath
                        filepath
                        filename)
                  (conj [] 
                        srcpath 
                        (if (seq group) (str/replace group "." "/"))
                        name
                        filename))]
    (->> pathvec
         (filter seq)
         (interpose "/")
         (apply str))))

(defn- write-to-version-file [func task-name project args]
  (let [{:keys [group name git-version]} project
        config (merge default-keys git-version)
        filepath (:filepath config)
        nsvec (if (seq filepath)
                (conj [] (str/replace filepath "/" ".") "version")
                (conj [] group name "version"))
        namespace (->> nsvec
                       (filter seq)
                       (interpose ".")
                       (apply str))
        code [";; Do not edit.  Generated by lein-git-version plugin."
              (str "(ns " namespace ")")
              ""
              (str "(def version \"" (get-file-version config) "\")")
              ""]]
    (spit (version-file project) (str/join "\n" code))
    (func task-name project args)))

(defn middleware
  [{:keys [git-version] :as project}]
  (let [config (merge default-keys git-version)
        version (get-version config)]
    (reduce #(assoc-in %1 %2 version) project (:assoc-keys config))))

(defn hooks []
  (add-hook #'leiningen.core.main/apply-task #'write-to-version-file))
