(ns leiningen.git-version
  (:require [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.java.shell :refer [sh]]
            [clojure.pprint :refer [pprint]]
            [clojure.string :as str :refer [trim]]
            [clj-time.core :as ct]
            [clj-time.format :as ctfmt]))

(defn- version-file [{:keys [git-version group name source-paths]}]
  (let [group (str/replace group "-" "_")
        name (str/replace name "-" "_")
        srcpath (first (filter seq (map #(re-find #".*src$" %) source-paths)))
        filename (:filename git-version)
        filepath (:filepath git-version)
        pathvec (if (seq filepath)
                  (conj [] 
                        srcpath
                        filepath
                        filename)
                  (conj [] 
                        srcpath 
                        (if (seq group) (str/replace group "." "/"))
                        name
                        filename))]
    (->> pathvec
         (filter seq)
         (interpose "/")
         (apply str))))

(defn- file-ns [group name filepath]
  (let [nsvec (if (seq filepath)
                (conj [] (str/replace (str/replace filepath "/" ".") "_" "-") "version")
                (conj [] group name "version"))]
    (->> nsvec
         (filter seq)
         (interpose ".")
         (apply str))))
  
(defn- write-to-version-file [project filever]
  (let [{:keys [group name git-version]} project
        filepath (:filepath git-version)
        code [";; Do not edit.  Generated by lein-git-version plugin."
              (str "(ns " (file-ns group name filepath) ")")
              ""
              (str "(def version \"" filever "\")")
              ""]]
    (spit (version-file project) (str/join "\n" code))))

(def ^:private write-file (memoize write-to-version-file))

(def ^:private date-formatter
  (-> :basic-date-time-no-ms 
      ctfmt/formatters
      (ctfmt/with-zone (ct/default-time-zone))))

(defn- gen-format-args [fmt argmap]
  (-> (for [v (rest fmt)]
        (if (keyword? v)
          (if (= :date v)
            (ctfmt/unparse date-formatter (ct/now))
            (v argmap))
          v))
      (conj (first fmt))))

(defn- get-version [version]
  (fn [cmd fmt]
    (if (seq cmd)
      (->> (apply sh cmd)
           :out
           str/trim
           (assoc {} :basever version :cmdver)
           (gen-format-args fmt)
           (apply format))
      version)))

(defn gen-version [{:keys [git-version version] :as project}]
  (let [{:keys [version-command version-file-command project-format file-format]} git-version
        getver (get-version version)
        jarver (getver version-command project-format)
        filever (getver version-file-command file-format)]
    (write-file project filever)
    jarver))

(defn git-version
  "Show git project version"
  [{:keys [group name git-version] :as project} & args]
  (load-file (version-file project))
  (println "Project Version:" (:version project))
  (println "File Version:   " (eval (symbol (file-ns group name (:filepath git-version)) "version"))))
