(ns leiningen.git-version
  (:require [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.java.shell :refer [sh]]
            [clojure.pprint :refer [pprint]]
            [clojure.string :as str :refer [trim]]
            [clj-time.core :as ct]
            [clj-time.format :as ctfmt]
            [org.ozias.cljlibs.logging.logging :refer (warnc)]))

(defn- version-file [{:keys [group name source-paths]}
                     {:keys [filename filepath]}]
  (let [group (str/replace group "-" "_")
        name (str/replace name "-" "_")
        srcpath (first (filter seq (map #(re-find #".*src$" %) source-paths)))
        pathvec (if (seq filepath)
                  (conj [] 
                        srcpath
                        filepath
                        filename)
                  (conj [] 
                        srcpath 
                        (if (seq group) (str/replace group "." "/"))
                        name
                        filename))]
    (->> pathvec
         (filter seq)
         (interpose "/")
         (apply str))))

(defn- file-ns [group name filepath]
  (let [nsvec (if (seq filepath)
                (conj [] (str/replace (str/replace filepath "/" ".") "_" "-") "version")
                (conj [] group name "version"))]
    (->> nsvec
         (filter seq)
         (interpose ".")
         (apply str))))
  
(defn- write-to-version-file [{:keys [group name] :as project}
                              {:keys [filepath] :as config}
                              version]
  (let [{:keys [group name]} project
        code [";; Do not edit.  Generated by lein-git-version plugin."
              (str "(ns " (file-ns group name filepath) ")")
              ""
              (str "(def version \"" version "\")")
              ""]]
    (spit (version-file project config) (str/join "\n" code))))

(def ^:private write-file (memoize write-to-version-file))

(defn- gen-format-args [fmt argmap]
  (-> (for [v (rest fmt)]
        (if (keyword? v)  
          (v argmap)
          v))
      (conj (first fmt))))

(def ^:private date-formatter
  (-> :basic-date-time-no-ms 
      ctfmt/formatters
      (ctfmt/with-zone (ct/default-time-zone))))

(defn- get-version [version]
  (fn [cmd fmt]
    (let [cmdver (if (seq cmd) (->> (apply sh cmd) :out str/trim) version)]
      (->> (ctfmt/unparse date-formatter (ct/now))
           (assoc {} :basever version :cmdver cmdver :date)
           (gen-format-args fmt)
           (apply format)))))

(defn- gen-version [{:keys [cmd fmt filename] :as config} {:keys [version] :as project}]
  (let [getver (get-version version)
        genver (getver cmd fmt)]
    (if (seq filename)
      (write-to-version-file project config genver))
    genver))
     
(defn gen-versions [{:keys [git-version version] :as project}]
  (let [getver (get-version version)]
    (for [[k config] git-version]
      {k (assoc config :version (gen-version config project))})))

(defn git-version
  "Show git project version"
  [{:keys [group name git-version] :as project} & args]
  (doseq [[k v] git-version]
    (warnc :green (str (clojure.core/name k) " Version:")
           :reset " "
           :cyan (:version v))))
