(ns pinkgorilla.notebook-ui.codemirror.key-binding
  (:require
   [taoensso.timbre :refer-macros [debug info infof error]]
   [re-frame.core :refer [dispatch]]
   [keybind.core :as key]
   [pinkgorilla.notebook-ui.codemirror.options :refer [code-mirror-key-down code-mirror-key-up]]))

(defn dispatch-buffer [cm-opts [fun-kw & args]]
  (info "running " fun-kw)
  (info "cm opts: " cm-opts)
  (info "args: " args)
  (if-let [f (fun-kw cm-opts)]
    (apply f args)
    (error "could not find function: " fun-kw)))

(defn bind!
  "Binds a sequence of button presses, specified by `spec`, to `cb` when
  pressed. Keys must be unique per `spec`, and can be used to remove keybinding
  with `unbind!`.
  `spec` format is emacs-like strings a-la \"ctrl-c k\", \"meta-shift-k\", etc."
  [BINDINGS {:keys [scope key spec handler]}]
  (infof "code-mirror [%s] key: [%s] event: %s" key spec handler)
  (swap! BINDINGS
         key/bind spec key (fn [{:keys [cm-opts e]}]
                             (info "fire!" e)
                             (case scope
                               :global (dispatch handler)
                               :current-buffer (dispatch-buffer cm-opts handler))
                             (.preventDefault e))))

(defn init-keybindings! [BINDINGS keybindings]
  (info "binding... " (count keybindings))
  (doall (for [b keybindings]
           (bind! BINDINGS b)))
  (info "binding done."))

(def DOWN (atom {}))
(def UP (atom {}))

(info "codemirror-down")
(init-keybindings! DOWN code-mirror-key-down)

;(info "codemirror-up")
;(init-keybindings! UP code-mirror-key-up)

(defn my-e->chord [{:keys [e]}]
  (info "my-e->chord")
  (into {} (for [[key attr] key/KEYATTRS]
             [key (aget e attr)])))

(defn dispatch! [bindings {:keys [e] :as args}]
  (when (get key/KNOWN-KEYS (.-keyCode e))
    (info "dispatching key: " (.-keyCode e))
    (with-redefs [key/e->chord my-e->chord]
      (let [r (key/dispatch args @bindings)]
        (info "r: " r)))))

; on-key-down in codemirror gets partially applied with cm-opts
; code-mirror raises normal js events [sender evt]
(defn on-key-down [cm-opts _ #_cm evt]
  (let [args {:cm-opts cm-opts :e evt}]
    (info "key-down!! " args "key: " (.-keyCode evt))
    (dispatch! DOWN args)))

(defn on-key-up [cm-opts evt]
  (info "key-up!! " (.-keyCode evt))
  (let [h (dispatch! UP {:context cm-opts :e evt})]
    (info "h: " h)
 ;    (repl-hint cm)
    ))


;; http://gcctech.org/csc/javascript/javascript_keycodes.htm


(def tab           9)
(def enter        13)
(def escape       27)

(def arrow-left   37)
(def arrow-up 	  38)
(def arrow-right  39)
(def arrow-down   40)

(def shift 	      16)
(def ctrl 	      17)
(def alt 	      18)
(def pause-break  19)
(def caps-lock 	  20)
(def window-left  91)
(def window-right 92)
(def select       93)

#_(defn on-key-down [cm-opts cm evt]
    (info "key-down" (.-keyCode evt))
    (case (.-keyCode evt)

      enter (when (and (not (.-shiftKey evt))
                       (.-metaKey evt))
              (when-let [source (.getValue cm)]
                (when (should-eval cm evt source)
                  (.preventDefault evt)
                  (run-eval source))))

      #'arrow-up (when (and (not (.-shiftKey evt))
                            (first-line? cm))
                   (.preventDefault evt)
                   (info "move-up")
                   (when-let [move-up (:move-up cm-opts)]
                     (move-up)))

      ~arrow-down (when (and (not (.-shiftKey evt))
                             (last-line? cm))
                    (info "last-line!!")
                    (.preventDefault evt)
                    (when-let [move-down (:move-down cm-opts)]
                      (move-down)))

      :none))

