(ns pinkgorilla.notebook-ui.keybindings.events
  (:require
   [clojure.string :as str]
   [taoensso.timbre :refer-macros [debug info error]]
   [re-frame.core :refer [reg-event-db reg-sub dispatch]]
   [webly.user.notifications.core :refer [add-notification]]
   [pinkgorilla.notebook-ui.keybindings.setup :refer [init-keybindings!]]
   [pinkgorilla.notebook-ui.keybindings.component :refer [keybindings-dialog]]))

(def clean-palette
  {:highlight     0
   :visible-items nil ;(:all-visible-commands palette)
   :query        ""})

(reg-event-db
 :keybindings/init
 (fn [db [_ keybindings]]
   (let [db (or db {})]
     (init-keybindings! keybindings)
     (assoc-in db
               [:keybindings] {:items keybindings
                               :palette clean-palette}))))

; dialog visibility

(reg-event-db
 :palette/show
 (fn [db [_]]
   (info "showing keybindings dialog")
   (dispatch [:palette/filter-changed ""])
   (dispatch [:modal/open [keybindings-dialog] :medium])
   (assoc-in db [:keybindings :palette] clean-palette)))

(reg-event-db
 :palette/hide
 (fn [db _]
   (dispatch [:modal/close])
   db))

; palette

(reg-sub
 :palette
 (fn [db _]
   (get-in db [:keybindings :palette])))

(defn text-matches-re
  [val item]
  (let [res (str/join ".*" (str/split (str/lower-case val) #""))
        re (re-pattern (str res ".*"))]
    (re-matches re (str/lower-case (:desc item)))))

(reg-event-db
 :palette/filter-changed
 (fn [db [_ query]]
   (let [keybindings (get-in db [:keybindings :items])
         palette (get-in db [:keybindings :palette])]
     (assoc-in db
               [:keybindings :palette]
               (merge palette {:visible-items (->> keybindings
                                                   (filter (partial text-matches-re query)))
                               :query        query})))))

(defn highlight-move [db direction]
  (let [palette (get-in db [:keybindings :palette])
        {:keys [highlight visible-items]} palette
        maxidx (- (count visible-items) 1)
        highlight-new (case direction
                        :up (if (> highlight 0)
                              (- highlight 1)
                              highlight)
                        :down (if (< highlight maxidx)
                                (+ highlight 1)
                                highlight))]
    (debug "highlight: " highlight-new)
    (assoc-in db [:keybindings :palette :highlight] highlight-new)))

(defn highlight-action [db]
  (let [palette (get-in db [:keybindings :palette])
        {:keys [highlight visible-items]} palette
        _ (info "action highlight: " highlight " count:" (count visible-items))
        item (when
              (not-empty visible-items)
               (nth visible-items highlight))]
    (dispatch [:palette/action item])
    db))

(reg-event-db
 :palette/filter-keydown
 (fn [db [_ keycode]]
   (case keycode
     38 (highlight-move db :up)
     40 (highlight-move db :down)
     27 (do (dispatch [:palette/hide])
            db)
     13 (highlight-action db)
     db)))

(reg-event-db
 :palette/action
 (fn [db [_ item]]
   (info "palette/action!" item)
   (let [handler (:handler item)]
     (if handler
       (do (info "dispatching" handler)
           (dispatch [:palette/hide]) ; if dispatch opens another dialog we first have to hide it
           (dispatch handler))
       (do
         (error "handler not found!")
         (add-notification :danger "keybining didn't have a handler!")
         (dispatch [:palette/hide])))
     db)))

