(ns pinkgorilla.notebook-ui.codemirror.editor
  (:require
   [taoensso.timbre :refer-macros [debug info]]
   [reagent.core :as r]
   [reagent.dom :as rd]
   [re-frame.core :refer [dispatch subscribe]]
   ["codemirror" :as CodeMirror]

   ["codemirror/addon/edit/closebrackets"]
   ["codemirror/addon/edit/matchbrackets"]
   ["codemirror/addon/hint/show-hint"]
   ["codemirror/addon/runmode/runmode"]
   ["codemirror/addon/runmode/colorize"]

   ["codemirror/mode/clojure/clojure"]
   ["codemirror/mode/markdown/markdown"]
   ; [cljsjs.codemirror.mode.xml]
   ; [cljsjs.codemirror.mode.javascript]
   ; ["parinfer-codemirror"]
   ; [cljsjs.codemirror.mode.clojure-parinfer]

   ;["codemirror/keymap/vim"]
   [pinkgorilla.ui.config :refer [link-css]]

   [pinkgorilla.notebook-ui.codemirror.eval :refer [should-eval run-eval]]
   [pinkgorilla.notebook-ui.codemirror.completion :refer [repl-hint]]
   [pinkgorilla.notebook-ui.codemirror.highlight]
   [pinkgorilla.notebook-ui.codemirror.buffer :refer [go-up? go-down?]]
   [pinkgorilla.notebook-ui.codemirror.parinfer :refer [on-cm-init]]
   [pinkgorilla.notebook-ui.codemirror.options :refer [cm-default-opts]]))

;; http://gcctech.org/csc/javascript/javascript_keycodes.htm

(def tab           9)
(def enter        13)
(def escape       27)

(def arrow-left   37)
(def arrow-up 	  38)
(def arrow-right  39)
(def arrow-down   40)

(def shift 	      16)
(def ctrl 	      17)
(def alt 	        18)
(def pause-break  19)
(def caps-lock 	  20)
(def window-left  91)
(def window-right 92)
(def select       93)

;; On Escape, should we revert to the pre-completion-text?
(def keys-complete-cancel #{enter escape})
(def keys-complete-nav #{tab shift})
(def keys-complete-show-all #{ctrl alt window-left select})

(defn on-key-down [cm-opts inst evt]
  (case (.-keyCode evt)

    (ctrl alt window-left select)
    (dispatch [:completions/show-all true])

    tab   (if (.-shiftKey evt)
            (dispatch [:completions/prior])
            (dispatch [:completions/next]))

    enter (when (and (not (.-shiftKey evt))
                     (.-metaKey evt))
            (when-let [source (.getValue inst)]
              (when (should-eval inst evt source))
              (.preventDefault evt)
              (run-eval source)))

    up     (let [source (.getValue inst)]
             (when (and (not (.-shiftKey evt))
                        (go-up? inst source))
               (.preventDefault evt)
               (when-let [move-up (:move-up cm-opts)]
                 (move-up))))

    down    (let [source (.getValue inst)]
              (when (and (not (.-shiftKey evt))
                         (go-down? inst source))
                (.preventDefault evt)
                (when-let [move-down (:move-down cm-opts)]
                  (move-down))))

    :none))

(defn on-key-up [inst evt]
  (let [key-code (.-keyCode evt)]
    (cond
      (keys-complete-cancel key-code) (dispatch [:completions/clear])
      (keys-complete-nav key-code) false
      (keys-complete-show-all key-code) (dispatch [:completions/show-all false])
      :else (repl-hint inst evt))))

(defn move-to-last-line [cm]
  (let [last-line (.lastLine cm)
        last-ch (count (.getLine cm last-line))]
    (.setCursor cm last-line last-ch)))

(defn on-mousedown [cm evt]
  (debug "on-mousedown"))

(defn focus-cm!
  [cm]
  (when cm
    (.focus cm)))

(defn save-code [cm buffer]
  (let [editor-code (.getValue cm)
        {:keys [code id]} buffer]
    (when-not (= code editor-code)
      (dispatch [:buffer/set-code id editor-code])
      true)))

(defn code-mirror-reagent
  "Create a code-mirror editor that knows a fair amount about being a good
  repl. The parameters:
  :style (reagent style map) will be applied to the container element"
  [buffer {:keys [style cm-opts]}]
  (let [cm (atom nil)]
    (r/create-class
     {:component-did-mount
      (fn [this]
        (let [el (rd/dom-node this)
              opts (clj->js
                    (merge
                     {:value (:code @buffer)}
                     cm-default-opts
                     cm-opts))
              _ (info "creating cm for: " el " opts: " opts)
              inst (CodeMirror. el opts)]
          (reset! cm inst)
          (.setValue inst (:code @buffer))
          (.on inst "change" (fn [] (save-code inst @buffer))
               (.on inst "keyup"  on-key-up)
               (.on inst "keydown" (partial on-key-down cm-opts))
               (.on inst "mousedown" on-mousedown)
          ;(when on-cm-init (on-cm-init inst))
          ;(focus-cm! inst)
               )))

      :component-did-update
      (fn [this old-argv]
        (focus-cm! @cm)
        (when (save-code @cm @buffer)
          (move-to-last-line @cm)))

      :reagent-render
      (fn [_ _ _]
        @buffer
        ;[:pre {:class "static-code clojure"}]
        [:div {:style style}])})))

(defn code-mirror [buffer-id options]
  (let [buffer (subscribe [:buffer/view buffer-id])]
    (fn [buffer-id options]
      [:div.w-full.h-full.bg-red-200 ; if this is :<> then it can fuck up flexbox styling
     ; dynamically loading css will not work the first time
     ; the editor is shown. css might get rendered after codemirror. 
       [link-css "codemirror/lib/codemirror.css"]
       [code-mirror-reagent buffer options]])))

