(ns pinkgorilla.nrepl.middleware.sniffer
  " observes a ide nrepl connection
    sends evals to a sink (typiccally pinkgorilla notebook)
   "
  (:require
   ;[clojure.tools.logging :refer (info)]
   [nrepl.transport :as transport]
   [nrepl.middleware :as middleware]
   [nrepl.middleware.print]
   [nrepl.misc :refer [response-for] :as misc]
   [pinkgorilla.nrepl.logger :refer [log]]
   [pinkgorilla.nrepl.middleware.picasso :refer [add-picasso]])
  (:import nrepl.transport.Transport))

(def state (atom {:msg-sink nil
                  :session-id-sink nil
                  :session-id-source nil}))

(defn session-id-
  "extracts the id from a session.
   sometimes sessions are strings, but sometimes 
   they are not."
  [session]
  (when session
    (if (instance? clojure.lang.AReference session)
      (-> session meta :id)
      session)))

(defn status- []
  (dissoc @state :msg-sink))

(defn response-sniff-status
  [req]
  (response-for req {:status :done
                     :sniffer-status (status-)}))

(defn sniff-on [{:keys [session] :as req}]
  (let [session (session-id- session)]
    ;(println "sniffer - setting source session id:" session)
    (swap! state assoc :session-id-source session)))

(defn sniff-off []
  (swap! state assoc :session-id-source nil))

(defn response-sniff-source!
  [req]
  (sniff-on req)
  (println "sniffer - state:" (status-))
  (response-for req {:status :done
                     :sniffer-status (status-)}))

(defn response-sniff-sink!
  "registers a sink, so sniffer knows to which nrepl-session
   to send the sniffed evals to"
  [{:keys [session] :as req}]
  (let [session (session-id- session)]
    (println "sink msg:" req)
    (println "sniffer - setting sink session id:" session)
    (swap! state assoc
           :msg-sink req
           :session-id-sink session)
    (println "sniffer - state:" (status-))
    ; response-msg may NOT contain :status :done
    ; sniffer will forward messages to that request-id
    (response-for req {; :status :done 
                       :sniffer-status (status-)})))

(defn res-eval-forward
  "forwards an nrepl message to the sink."
  [req]
  (let [req-listener (:msg-sink @state)
        req-forward (dissoc req :session :transport
                            :nrepl.middleware.print/keys
                            :nrepl.middleware.print/print-fn
                            :nrepl.middleware.caught/caught-fn)
        req-send (response-for req-listener {;:status :done
                                             :sniffer-forward req-forward})]
    (println "sniffer forwarding to " (:session-id-sink @state) "message: " req-forward)
    (when (:code req)
      (println "code meta data: " (meta (:code req))))
    req-send))


; a clean nrepl middleware is found in:
; https://github.com/RickMoynihan/nrebl.middleware/blob/master/src/nrebl/middleware.clj


(defn commands [req res]
  (let [v (:value res)]
    (when v
      (case v
        :gorilla/on (do (log "enabling sniffing.")
                        (sniff-on res))
        :gorilla/off (do (log "disabling sniffing.")
                         (sniff-off))
        nil))))

(defn eval-res [{:keys [code] :as req} {:keys [value] :as res}]
  (when (and code true); (contains? res :value))
    (let [msg-listener (:msg-sink @state)
          res-forward (dissoc res :session :transport
                              :nrepl.middleware.print/print-fn
                              :nrepl.middleware.caught/caught-fn)
          res-forward (if (:as-picasso res-forward)
                        res-forward
                        (add-picasso res-forward))
          res-resp (response-for msg-listener {:sniffer-forward res-forward})]
      ; printing not allowed here - nrepl would capture this as part of the eval request 
      ;(println "sniffer forwarding response:" msg-resp)
      res-resp)))

(defn- wrap-sniffer-sender
  "Wraps a `Transport` with code which prints the value of messages sent to
  it using the provided function."
  [{:keys [id op ^Transport transport session] :as req}]
  (reify Transport
    (recv [this]
      (.recv transport))
    (recv [this timeout]
      (.recv transport timeout))
    (send [this res]
      (.send transport res)
      (commands req res)
      (when (and (= (session-id- session) (:session-id-source @state))
                 (:code req))
        (when (:msg-sink @state)
          (transport/send (:transport (:msg-sink @state))
                          (eval-res req res))))
      this)))

(defn wrap-sniffer
  [handler]
  (fn [{:keys [^Transport transport op session] :as req}]
    (let [session (session-id- session)]
      (cond
        ; requests handled by sniffer don't have to be processed by other handers
        (= op "sniffer-status")
        (transport/send transport (response-sniff-status req))

        (= op "sniffer-source")
        (transport/send transport (response-sniff-source! req))

        (= op "sniffer-sink")
        (transport/send transport (response-sniff-sink! req))

        :else
        (do (when (and (= op "eval")
                       (= session (:session-id-source @state)))
              (println "sniffer - forwarding eval: " (:code req))
              (if (:msg-sink @state)
                (transport/send (:transport (:msg-sink @state)) (res-eval-forward req))
                (println "sniffer - no sink. cannot forward!"))
                  ;(handler request)
              )
            (handler (assoc req :transport (wrap-sniffer-sender req))))
        ;  (handler request)        
        ))))


; https://nrepl.org/nrepl/design/middleware.html


(middleware/set-descriptor!
 #'wrap-sniffer
 {:requires #{}
  :expects  ; expects get executed before this middleware
  #{"eval"
    #'pinkgorilla.nrepl.middleware.picasso/wrap-picasso}
  :handles {"sniffer-status"
            {:doc "status of sniffer middleware"}

            "sniffer-source"
            {:doc "start sniffing current session"}

            "sniffer-sink"
            {:doc "called from notebook. destination for forwarded events "}}})



; {:doc "Provides sniffer status"
;             :requires {"prefix" "The prefix to complete."}
;             :optional {"ns" "The namespace in which we want to obtain completion candidates. Defaults to `*ns*`."
;                        "complete-fn" "The fully qualified name of a completion function to use instead of the default one (e.g. `my.ns/completion`)."
;                        "options" "A map of options supported by the completion function."}
;             :returns {"completions" "A list of completion candidates. Each candidate is a map with `:candidate` and `:type` keys. Vars also have a `:ns` key."}}}})

