(ns jdbc.crud
  (:refer-clojure :exclude [find])
  (:require [clojure.tools.logging :as log]
            [clojure.java.jdbc :as jdbc]
            [honeysql.core :as sql]
            [honeysql.helpers :as helpers]
            [jdbc.util :as util]))

(defn query
  "execute a honeysql map OR a clojure.java.jdbc sql vector"
  ([db sql-map params]
   (query db (sql/format sql-map params)))
  ([db sql-v]
   (jdbc/query db sql-v)))

(defn execute!
  "execute a honeysql map OR a clojure.java.jdbc sql vector"
  ([db sql-map params]
   (execute! db (sql/format sql-map params)))
  ([db sql-v]
   (jdbc/execute! db sql-v)))

(defn select
  "select from table based on a sequence of criteria"
  [db table & criteria]
  (let [sql-v (sql/format (apply util/select-map table criteria))]
    (log/debug sql-v)
    (query db sql-v)))

(defn find [db table & criteria]
  (first (apply select db table criteria)))

(defn find-by-id [db table id]
  (find db table {:id id}))

(defn find-all-by [db table attr value]
  (select db table [attr value]))

(defn find-by [db table attr value]
  (find db table [attr value]))

(defn exists?
  "check existence in table based on a sequence of criteria"
  [db table & criteria]
  (let [sql-v (sql/format (apply util/exists-map table criteria))]
    (log/debug sql-v)
    (let [{:keys [n]} (or (first (query db sql-v)) {:n 0})]
      (if (> n 0) n))))

(defn- generated-key [result]
  (or (:generated_key result)
      ((keyword "scope_identity()") result)
      (:id result)
      ((keyword "last_insert_rowid()") result)))

(defn insert!
  "Insert a new record in the database with fields from map m.
  Returns the id generated by jdbc, or, if there is none (for cases
  when :id is pre-supplied), returns :id from m."
  [db table m]
  (let [result (first (jdbc/insert! db table m))]
    (generated-key (merge m result))))

(defn create!
  "Inserts a record in the database with fields from map m
   and returns a new map of the created record. This operates
   on the database within a transaction."
  [db table m]
  (jdbc/with-db-transaction [tx db]
    (find-by-id tx table (insert! tx table m))))

(defn- update* [db table m & criteria]
  (if (empty? m)
    (throw (IllegalArgumentException. "no values to set on update"))
    (let [sql-v (sql/format (apply util/update-map table m criteria))]
      (log/debug sql-v)
      (execute! db sql-v))))

(defn update!
  "if m has an :id update using critera including [:= :id id]
     use m as the set values for the rec in a transaction

   else if we have only criteria and no id in m
      use m as the set values for ANY recs matching criteria

   else use m as set values to update entire db"
  [db table m & criteria]
  (if-let [id (get m :id)]
    (let [criteria (cons {:id id} criteria)]
      (jdbc/with-db-transaction [tx db]
        (apply update* tx table (dissoc m :id) criteria)
        (find-by-id tx table id)))
    (apply update* db table m criteria)))

(defn delete!
  "delete according to a set of criteria"
  [db table & criteria]
  (if criteria
    (let [sql-v (sql/format (apply util/delete-map table criteria))]
      (log/debug sql-v)
      (execute! db sql-v))
    (throw (IllegalArgumentException. "no criteria found for delete"))))

(defn delete-by-id!
  "Deletes the entity with the given id from the table in db."
  [db table id]
  (first (jdbc/delete! db table ["id = ?" id])))

