; This Source Code Form is subject to the terms of the Mozilla Public License,
; v. 2.0. If a copy of the MPL was not distributed with this file, You can
; obtain one at http://mozilla.org/MPL/2.0/.

(ns quilt.sedan
  (:require [quilt.sedan.impl :as impl]
            [quilt.sedan.sbuffers :as sbuffers])
  (:refer-clojure :exclude (compare)))

(defn encode
  "Produce the sortable encoded representation of [x], and either return it, or
  write it to [buffer], depending on the arity invoked.  If provided, the latter
  must be an instance of java.lang.Appendable on the JVM, or an object that has
  an `append(String)` method in JavaScript."
  ([x] (let [b (sbuffers/create)]
         (encode x b)
         (str b)))
  ([x buffer] (impl/encode x buffer)))

(defn decode
  "Decode a single value from the given string."
  [s]
  (impl/decode s))

(defn compare
  "Compare [x] and [y]. Manifests the collation semantics of Sedan at runtime,
  without serializing anything. Values of differing types may be safely
  compared, as long as those types are supported by Sedan."
  [x y]
  (impl/compare* x y))

(def bottom
  "An instance of a value that is guaranteed to sort before all other values,
  either when `encode`d, or when `compare`d.  Provided to avoid sillyness like
  having a string of ten NUL characters around to (hopefully) establish a lower
  bound."
  (impl/->Bottom))

(def top
  "An instance of a value that is guaranteed to sort after all other values,
  either when `encode`d, or when `compare`d.  Provided to avoid sillyness like
  having a string of ten 'z', DEL, or FFF0 characters around to (hopefully)
  establish an upper bound."
  (impl/->Top))



;;;;;;;;;;;; This file autogenerated from src/cljx/quilt/sedan.cljx
