(ns rksm.cloxp-projects.tasks
  (:require [clojure.java.shell :as shell]
            [leiningen.core.main :as lmain]
            [leiningen.core.project :as lproj]
            [clojure.java.io :as io]
            [clojure.string :refer [join split-lines]]))

(defn shell
  [{:keys [dir] :as project} & cmd+args]
  (shell/with-sh-dir dir
    {:project project :result (apply shell/sh cmd+args)}))

(defn shell-for-all
  [projects & cmd+args]
  (map #(apply shell % cmd+args) projects))

(defn- format-shell-result-for-print
  [{:keys [project result]}]
  (format "%s:\n%s"
          (:dir project)
          (->> result
            ((juxt :out :err))
            (join "\n")
            split-lines
            (map (partial str "  "))
            (join "\n"))))

(defn- format-shell-results-for-print
  [results]
  (join 
   "\n-=-=-=-\n"
   (map format-shell-result-for-print results)))

(defn print-shell-for-all
  [projects & cmd+args]
  (print (format-shell-results-for-print
          (apply shell-for-all projects cmd+args))))

; -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; git

(defn git-status-for-all
  [projects]
  (let [commands
        "[[ ! -d .git ]] && exit 0;
        branch=$(git branch | grep \"*\" | cut -d \" \" -f 2);
        local_hash=$(git show-ref --hash $branch | tail -n 1);
        remote_hash=$(git ls-remote origin $branch | cut -f 1);
        if [[ $local_hash = $remote_hash ]]; then echo \"No remote updates\"; else echo \"Remote updates\"; fi;
        git status --porcelain;"]
    (print-shell-for-all projects "/bin/bash" "-c" commands)))

(defn git-short-status-for-all
  [projects]
  (print-shell-for-all projects "git" "status" "--porcelain"))

; -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
; leiningen

(defn lein-project-map
  [{:keys [dir] :as project}]
  (lproj/project-with-profiles
   (lproj/read
    (str (io/file dir "project.clj")))))

(defn install
  [project]
  (let [cmds (if (-> project lein-project-map :aliases (get "cleaninstall"))
               ["cleaninstall"] ["do" "clean," "install"])]
    (apply shell project "lein" cmds)))

(defn install-all
  [projects]
  (print (format-shell-results-for-print (map install projects))))

(defn deploy-clojars
  [project]
  (let [cmds (if (-> project lein-project-map :aliases (get "cleandeploy"))
               ["cleandeploy"] ["do" "clean," "deploy" "clojars"])]
    (apply shell project "lein" cmds)))

(defn deploy-clojars-all
  [projects]
  (print (format-shell-results-for-print (map deploy-clojars projects))))

(defn run-tests
  [project]
  (let [cmds (if (-> project lein-project-map :aliases (get "cleantest"))
               ["cleantest"] ["do" "clean," "test"])]
    (apply shell project "lein" cmds)))

(defn run-tests-all
  [projects]
  (print (format-shell-results-for-print (map run-tests projects))))

(defn lein
  [project & task+args]
  (binding [leiningen.core.main/*exit-process?* false
            leiningen.core.main/*debug* false
            leiningen.core.main/*info* false
            ;  *err* *out*
            ;  *out* (java.io.StringWriter.)
            ]
    (lmain/resolve-and-apply (lein-project-map project) task+args)
    #_(str *out*)))

; -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

(comment
 
 (require '[rksm.cloxp-projects.core :as projs])
 (require '[rksm.cloxp-projects.dependencies :as proj-deps])
 (require '[rksm.cloxp-projects.tasks :as tasks])
 
 (rksm.cloxp-projects.core/project-info
  "/Users/robert/clojure/cloxp-projects"
  {:additional-keys [:dev-dependencies]})
 
 (def projects (proj-deps/sort-by-deps
                (rksm.cloxp-projects.core/project-infos
                 ["/Users/robert/clojure/clojure-system-files"
                  "/Users/robert/clojure/cloxp"
                  "/Users/robert/clojure/cloxp-4clojure"
                  "/Users/robert/clojure/cloxp-cljs"
                  "/Users/robert/clojure/cloxp-cljs-repl"
                  "/Users/robert/clojure/cloxp-com"
                  "/Users/robert/clojure/cloxp-lein"
                  "/Users/robert/clojure/cloxp-projects"
                  "/Users/robert/clojure/cloxp-repl"
                  "/Users/robert/clojure/cloxp-server"
                  "/Users/robert/clojure/cloxp-source-reader"
                  "/Users/robert/clojure/cloxp-trace"
                  "/Users/robert/clojure/subprocess"
                  "/Users/robert/clojure/system-navigator"
                  "/Users/robert/clojure/cljs-slimerjs-tester"
                  "/Users/robert/clojure/test-helpers"]
                 {:include-plugins? true, :include-dev? true, :clean? true})))
 
 (def p (first (proj-deps/projects-matching projects "cloxp-com")))
 
 (tasks/git-short-status-for-all projects)
 (tasks/git-status-for-all projects)
 
 (tasks/install-all projects)
 
 (tasks/print-shell-for-all projects "lein" "install")
 (tasks/print-shell-for-all projects "lein" "ancient")

 )