(ns rksm.cloxp-projects.project-clj
  (:require [clojure.java.io :as io]
            [leiningen.core.project :as project]
            [rksm.system-files :as sf]
            [rksm.system-files.jar-util :refer [jar-url->reader jar?]]))

(defn- file-coerce
  [file-or-string]
  (try
    (io/file file-or-string)
    (catch Exception e nil)))

(defn- file?
  [file-or-string]
  (if-let [f (file-coerce file-or-string)]
    (and (.exists f)
         (-> f .isDirectory not))))

(defn- dir?
  [file-or-string]
  (if-let [f (file-coerce file-or-string)]
    (.isDirectory f)))

(defmulti read-conf
  (fn [x] (cond
            (jar? x) :jar
            (file? x) :file
            (dir? x) :dir
            (string? x) :string)))

(defmethod read-conf :string
  [project-clj-string]
  ; rk 2015-03-22: this is copied from leiningen.core.project/read-raw as the
  ; leiningen methods do not provide a non-file interface to reading project maps
  ; and jars aren't supported either
  (locking leiningen.core.project/read-raw
    (binding [*ns* (find-ns 'leiningen.core.project)]
      (try (eval (read-string project-clj-string))
        (catch Exception e
          (throw (Exception. "Error loading" e)))))
    (let [project (resolve 'leiningen.core.project/project)]
      (when-not project
        (throw (Exception. "No project map!")))
      ;; return it to original state
      (ns-unmap 'leiningen.core.project 'project)
      @project)))

(defmethod read-conf :jar
  [^java.io.File jar-file]
  (binding [*file* (str jar-file)] ; for leiningen's defproject
    (if-let [rdr (jar-url->reader
                   (str "jar:" (.toURI jar-file) "!/project.clj"))]
      (read-conf (slurp rdr)))))

(defmethod read-conf :dir
  [dir]
  (let [project-clj (io/file dir "project.clj")]
    (read-conf
     (if (.exists project-clj)
       project-clj
       (.getParentFile dir)))))

(defmethod read-conf :file
  [file]
  (binding [*file* (str file)]
    (read-conf (slurp *file*))))

(defmethod read-conf :default
  [x]
  nil)

(comment
 (dir? "/Users/robert/clojure/cloxp-projects")
 (read-conf (slurp "/Users/robert/clojure/cloxp-projects/project.clj"))
 (read-conf "/Users/robert/clojure/cloxp-projects/project.clj")
 (dir? "/Users/robert/clojure/cloxp-projects/")
 (read-conf "/Users/robert/clojure/cloxp-projects/")
 (read-conf (io/file "/Users/robert/clojure/cloxp-projects/target/cloxp-projects-0.1.5.jar"))
 )
; -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

(defn lein-project-conf-for-ns
  [ns]
  (read-conf
   (sf/classpath-for-ns ns)))
