(ns scrum.devtools.state-tree
  (:require [rum.core :as rum]
            [scrum.devtools.styles :as styles]))

(declare render)


(rum/defc -Syntax [ch]
  [:div {:style (styles/syntax-literal)}
   ch])


(rum/defc -Keyword [value]
  [:div {:style (styles/keyword-literal)}
   (str value)])

(rum/defc -Number [value]
  [:div {:style (styles/number-literal)}
   (str value)])

(rum/defc -String [value]
  [:div {:style (styles/string-literal)}
   [:em (js/JSON.stringify value)]])

(rum/defc -Boolean [value]
  [:div {:style (styles/boolean-literal)}
   (str value)])

(rum/defc -Nil [_]
  [:div {:style (styles/nil-literal)}
   "nil"])


(rum/defc -MapEntry
  [first? last? nested? {:keys [start end]} [key value]]
  (let [single-margin "7.5px"
        double-margin "15px"
        margin (if nested? double-margin single-margin)]
    [:div
     (cond
       (and first? nested? (nil? start))
       {:style (assoc (styles/map-entry-literal)
                 :marginLeft single-margin
                 :display "inline-block")}

       first?
       {:style (assoc (styles/map-entry-literal)
                 :display "inline-block")}

       last?
       {:style (assoc (styles/map-entry-literal)
                 :marginLeft margin
                 :display "inline-block")}

       :else
       {:style (assoc (styles/map-entry-literal)
                 :marginLeft margin)})
     (when first? start)
     (when first? (-Syntax "{"))
     (render key)
     [:span " "]
     (if (and (not (keyword? value))
              (not (number? value))
              (not (string? value))
              (not (boolean? value))
              (not (nil? value)))
       (if (> (count value) 1)
         [:div (render value)]
         (render value))
       [:div
        {:style {:display "inline-block"
                 :whiteSpace "nowrap"}}
        (render value)])
     (when last? (-Syntax "}"))
     (when last? end)]))

(rum/defc -Map
  ([value]
   (-Map value false {}))
  ([value nested? sntx]
   (let [vcount (dec (count value))]
     [:div
      (if nested?
        {:style (styles/map-literal)}
        {:style (assoc (styles/map-literal)
                  :display "inline-block")})
      (map-indexed
        #(-MapEntry (zero? %1) (= %1 vcount) nested? sntx %2)
        value)])))

(rum/defc -Vector [value]
  [:div {:style (styles/vector-literal)}
   (-Syntax "[")
   (map render value)
   (-Syntax "]")])

(rum/defc -List [value]
  (let [lcount (dec (count value))]
    [:div {:style (styles/list-literal)}
     (map-indexed
       #(render %2 true
                (cond
                  (zero? %1) {:start (-Syntax "(")}
                  (= %1 lcount) {:end (-Syntax ")")}
                  :else {}))
       value)]))


(defmulti render (fn [value] (type value)))

(defmethod render :default [value]
  (js/console.log (str ::render) (type value)))

(defmethod render cljs.core.Keyword [value]
  (-Keyword value))

(defmethod render js/Number [value]
  (-Number value))

(defmethod render js/String [value]
  (-String value))

(defmethod render js/Boolean [value]
  (-Boolean value))

(defmethod render nil [value]
  (-Nil value))


(defmethod render cljs.core.PersistentArrayMap [value nested? sntx]
  (-Map value nested? sntx))

(defmethod render cljs.core.PersistentHashMap [value nested? sntx]
  (-Map value nested? sntx))

(defmethod render cljs.core.PersistentVector [value]
  (-Vector value))

(defmethod render cljs.core.List [value]
  (-List value))
