;; Copyright (c) 2008,2009 Lau B. Jensen <lau.jensen {at} bestinclass.dk
;;                         Meikel Brandmeyer <mb {at} kotka.de>
;; All rights reserved.
;;
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file LICENSE.txt at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by the
;; terms of this license. You must not remove this notice, or any other, from
;; this software.

(clojure.core/ns clojureql.backend.derby
  (:import
     clojure.lang.RT)
  (:require
     [clojureql :as cql]
     [clojureql.util :as util]
     [clojureql.schema :as schema]))

(defn load-embedded-driver
  "Load the embedded derby driver."
  []
  (let [klass (RT/classForName "org.apache.derby.jdbc.EmbeddedDriver")]
    (.newInstance klass)
    nil))

(defn load-client-driver
  "Load the derby client driver."
  []
  (let [klass (RT/classForName "org.apache.derby.jdbc.ClientDriver")]
    (.newInstance klass)
    nil))

; Register for emulation of full join.
(swap! cql/sql-hierarchy
       derive org.apache.derby.iapi.jdbc.EngineConnection ::cql/EmulateFullJoin)

(defn derby-foreign-key [columns references]
  (let [[table ref-cols actions] (cql/compile-references references)
        actions (if (contains? #{'cascade 'set-null} (actions 'update))
                  (dissoc actions 'update)
                  actions)]
  (util/str-cat " "
    ["FOREIGN KEY" columns "REFERENCES" table (or ref-cols columns)
     (cql/triggered-action 'delete actions)
     (cql/triggered-action 'update actions)])))

(defmethod cql/compile-sql
  [::cql/CreateTable org.apache.derby.iapi.jdbc.EngineConnection]
  [stmt _]
  (cql/with-table-options stmt
    (str "CREATE TABLE " table " ("
      (util/str-cat ","
        (concat
          (map (fn [[column col-type]]
                 (str column " "
                      (cql/standard-column col-type
                        :not-null (contains? non-nulls column)
                        :default  (get defaults column)
                        :others   (cql/column-option "GENERATED BY DEFAULT AS IDENTITY" column auto-inc))))
               columns)
          [(cql/list-constraint "PRIMARY KEY" primary-key)]
          (map cql/standard-unique uniques)
          (map cql/standard-check checks)
          (map #(apply derby-foreign-key %) foreign-key)))
      ")")))

(defmethod cql/compile-sql
  [::cql/DropTable org.apache.derby.iapi.jdbc.EngineConnection]
  [stmt db]
  (let [{:keys [table if-exists]} stmt]
    (when (or (not if-exists)
              (schema/table db (.toUpperCase table)))
      (str "DROP TABLE " table))))

(defmethod cql/compile-sql
  [::cql/DropView org.apache.derby.iapi.jdbc.EngineConnection]
  [stmt db]
  (let [{:keys [view if-exists]} stmt]
    (when (or (not if-exists)
              (schema/view db (.toUpperCase view)))
      (str "DROP VIEW " view))))

(defmethod cql/compile-sql-alter
  [::cql/Rename org.apache.derby.iapi.jdbc.EngineConnection]
  [stmt _]
  (let [{:keys [options table]} stmt]
    (util/str-cat " "
      (cons "RENAME"
        (if (= 'column (first options))
          ["COLUMN" (str table "." (options 1)) "TO" (options 2)]
          ["TABLE"  table                       "TO" (first options)])))))
