;; (c) 2008,2009 Lau B. Jensen <lau.jensen {at} bestinclass.dk
;;                         Meikel Brandmeyer <mb {at} kotka.de>
;; All rights reserved.
;;
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file LICENSE.txt at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by the
;; terms of this license. You must not remove this notice, or any other, from
;; this software.

(clojure.core/in-ns 'clojureql)

;; STANDARD SQL HELPERS ====================================

(defn standard-primary-key
  "Returns a SQL primary key clause for the given arguments."
  [identifiers]
  (list-constraint "PRIMARY KEY" identifiers))

(defn standard-unique
  "Returns a SQL unique clause for the given arguments."
  [identifiers]
  (list-constraint "UNIQUE" identifiers))

(defn standard-check
  "Returns a SQL check clause for the given arguments."
  [condition]
  (str "CHECK " (compile-function condition)))

(def #^{:doc "Map of symbols to referencial actions."}
  referencial-actions
  {'cascade   "CASCADE"
   'no-action "NO ACTION"
   'restrict  "RESTRICT"
   'set-null  "SET NULL"})

(defn triggered-action
  "Returns a triggered referential action for use in foreign key
  references."
  [action actions]
  (when (contains? actions action)
    (str-cat " "
      ["ON" (.toUpperCase (str action))
       (-> action actions referencial-actions)])))

(defn compile-references
  "Compiles the references part of foreign key constraints."
  [references]
  (let [[table ref-cols & actions] references
        [ref-cols actions] (if (= 'on ref-cols)
                             [nil (cons ref-cols actions)]
                             [ref-cols actions])
        actions (apply hash-map (filter #(not (= 'on %)) actions))]
    [table ref-cols actions]))

(defn standard-foreign-key
  "Returns a standard compliant foreign key constraint given a
  column vector and reference vector."
  [columns references]
  (let [[table ref-cols actions] (compile-references references)
        columns  (->vector columns)
        ref-cols (when ref-cols (->vector ref-cols))]
    (str-cat " "
      ["FOREIGN KEY"
       (identifiers-list columns)
       "REFERENCES" table
       (identifiers-list (or ref-cols columns))
       (triggered-action 'delete actions)
       (triggered-action 'update actions)])))

(defn column-option
  "Returns a column option if the specified column is listed in the
  identifiers collection, else nil."
  ([name column identifiers]
     (column-option name column identifiers nil))
  ([name column identifiers args]
     (when (contains? identifiers column)
       (str name
         (when args
           (str " " (str-cat " " args)))))))

(defn standard-default
  "Returns a standard compliant default clause."
  [value]
  (str "DEFAULT " (compile-function value)))

(defn compile-column-type
  "Returns the column datatype, convert list expressions to parametrized
  datatypes."
  [col-type]
  (if (list? col-type)
    (str (first col-type) "(" (second col-type) ")")
    col-type))

(defn standard-column
  "Returns a standard compliant column definition clause. Supports
  not-null and default options, you could also pass more options as
  string with the :others option."
  [col-type & options]
  (let [options (apply hash-map options)
        {:keys [default not-null]} options]
    (str-cat " "
      (list (compile-column-type col-type)
            (when not-null "NOT NULL")
            (when default (standard-default default))
            (:others options)))))

;; CREATE ==================================================

(defn standard-table-constraints
  "Returns a list for all given table constraints."
  [primary-key uniques checks foreign-keys]
  (concat
    [(standard-primary-key primary-key)]
    (map standard-unique uniques)
    (map standard-check checks)
    (map #(apply standard-foreign-key %) foreign-keys)))

(defmacro with-table-options
  "Extracts all relevant variables from the given statement for create-table."
  [stmt & body]
  `(let [{:keys [~'table ~'columns ~'options]} ~stmt
         {:keys [~'checks ~'foreign-key ~'primary-key ~'uniques
                 ~'auto-inc ~'defaults ~'non-nulls]} ~'options
         ;; table constraints
         ~'checks      (->vector ~'checks)
         ~'foreign-key (apply hash-map ~'foreign-key)
         ~'primary-key (->vector ~'primary-key)
         ~'uniques     (map ->vector (->vector ~'uniques))
         ;; column options
         ~'auto-inc    (set (->vector ~'auto-inc))
         ~'defaults    (apply hash-map ~'defaults)
         ~'non-nulls   (set (->vector ~'non-nulls))]
     ~@body))

(defmethod compile-sql [::CreateTable ::Generic]
  [stmt _]
  (with-table-options stmt
    (str "CREATE TABLE " table " ("
         (str-cat ","
           (concat (map (fn [[column col-type]]
                          (str column " "
                               (standard-column col-type
                                 :not-null (contains? non-nulls column)
                                 :default  (get defaults column)
                                 :others   (column-option "GENERATED BY DEFAULT AS IDENTITY" column auto-inc))))
                        columns)
             (standard-table-constraints
               primary-key
               uniques
               checks
               foreign-key)))
      ")")))

(defmethod compile-sql [::CreateView ::Generic]
  [stmt conn]
  (let [{:keys [view query columns]} stmt]
    (str-cat " " ["CREATE VIEW" view
                  (str "(" (str-cat "," columns) ")")
                  "AS" (compile-sql query conn)])))

;; DROP ====================================================

(defmethod compile-sql [::DropTable ::Generic]
  [stmt _]
  (let [{:keys [table if-exists]} stmt]
    (str-cat " " ["DROP TABLE"
                  (when if-exists
                    "IF EXISTS")
                  table])))

(defmethod compile-sql [::DropView ::Generic]
  [stmt _]
  (let [{:keys [view if-exists]} stmt]
    (str-cat " " ["DROP VIEW"
                  (when if-exists
                    "IF EXISTS")
                  view])))

;; ALTER ===================================================

(defn standard-table-constraint
  "Returns a single table constraint for use in alter-table."
  [constraint options]
  (condp = constraint
    'unique      (standard-unique options)
    'primary-key (standard-primary-key options)
    'check       (standard-check (first options))
    'foreign-key (standard-foreign-key (first options) (rest options))))

(defmulti compile-sql-alter
  "Sub method to compile ALTER statements."
  {:arglists '([stmt db])}
  (fn [stmt db] [(stmt :action) (class db)])
  :hierarchy sql-hierarchy)

(defmethod compile-sql-alter [::Add ::Generic]
  [stmt _]
  (let [{:keys [options table]} stmt
        object  (first options)
        options (rest options)]
    (str-cat " "
      ["ALTER TABLE" table "ADD"
        (if (= object 'column)
          (str "COLUMN " (first options) " "
               (apply standard-column (rest options)))
          (standard-table-constraint object options))])))

(defmethod compile-sql-alter [::Drop ::Generic]
  [stmt _]
  (let [{:keys [options table]} stmt
        object  (first options)
        options (rest options)]
    (str-cat " "
      ["ALTER TABLE" table "DROP"
        (condp = object
          'column     "COLUMN"
          'constraint "CONSTRAINT")
        (first options)
        (condp = (second options)
          'cascade  "CASCADE"
          'restrict "RESTRICT")])))

(defn standard-alter-table-default
  "Returns standard compliant drop clause for alter-table."
  [action _ & value]
  (condp = action
    'drop "DROP DEFAULT"
    'set  (str "SET " (standard-default (first value)))))

(defmethod compile-sql-alter [::Modify ::Generic]
  [stmt _]
  (let [{:keys [options table]} stmt
        column  (first options)
        options (rest options)]
    (str-cat " "
      ["ALTER TABLE" table "ALTER COLUMN" column
        (apply standard-alter-table-default options)])))

(defmethod compile-sql-alter [::Rename ::Generic]
  [stmt _]
  (throw (Exception. "The SQL standard does not support renaming columns or constraints.")))

(defmethod compile-sql [::AlterTable ::Generic]
  [stmt db]
  (let [{:keys [action options table]} stmt]
    (str-cat ";"
      (map #(compile-sql-alter
              (assoc stmt :options %) db) options))))
