;; Copyright (c) 2008,2009 Lau B. Jensen <lau.jensen {at} bestinclass.dk
;;                         Meikel Brandmeyer <mb {at} kotka.de>
;; All rights reserved.
;;
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file LICENSE.txt at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by the
;; terms of this license. You must not remove this notice, or any other, from
;; this software.

(clojure.core/in-ns 'clojureql)

;; DEFINITIONS =============================================

(defn- trace*
  [qx x]
  (print "TRACE: ") (pr qx) (print " = ") (prn x) (flush)
  x)

(defmacro #^{:private true} trace
  [x]
  `(trace* (quote ~x) ~x))

(defn- make-type
  "Tag the given map with a :type tag in the metadata."
  [the-map typ]
  (with-meta the-map {:type typ}))

; Queries

(defstruct sql-query
  :columns :tables :predicates :column-aliases :table-aliases :env)

(defstruct sql-join
  :query :on :env)

(defstruct sql-ordered-query
  :query :order :columns :env)

(defstruct sql-grouped-query
  :query :columns :env)

(defstruct sql-having-query
  :query :env)

(defstruct sql-distinct-query
  :query :env)

(defstruct sql-union
  :all :queries :env)

(defstruct sql-intersect
  :queries :env)

(defstruct sql-difference
  :queries :env)

(defstruct sql-let-query
  :fn)

; Data Handling

(defstruct sql-insert-query
  :table :query)

(defstruct sql-insert-values
  :table :columns :env)

(defstruct sql-update
  :table :columns :predicates :env)

(defstruct sql-delete
  :table :predicates :env)

; Table Handling

(defstruct sql-create-table
  :table :columns)

(defstruct sql-create-view
  :view :query :columns)

(defstruct sql-drop-table
  :table :if-exists)

(defstruct sql-drop-view
  :view)

(defstruct sql-alter-table
  :table :action)

; Special Statements

(defstruct sql-batch-statement
  :statements)

(defstruct sql-raw-statement
  :statement)

;; HIERARCHY ===============================================

(def
  #^{:private true
     :doc
  "Hierarchy for the SELECT statements."}
  select-hierarchy
  (-> (make-hierarchy)
    (derive ::Join           ::Select)
    (derive ::InnerJoin      ::Join)
    (derive ::LeftJoin       ::Join)
    (derive ::RightJoin      ::Join)
    (derive ::FullJoin       ::Join)
    (derive ::OrderedSelect  ::Select)
    (derive ::GroupedSelect  ::Select)
    (derive ::DistinctSelect ::Select)))

(defn- is-and-not?
  "Checks whether the given query is of the type isa (or any derivee).
  If so, checks also any subquery. Returns nil or the offending query
  type."
  [kwery isa is-not-a]
  (if (and      (isa? select-hierarchy (type kwery) isa)
           (not (isa? select-hierarchy (type kwery) is-not-a)))
    (when-let [sub-query (kwery :query)]
      (is-and-not? sub-query isa is-not-a))
    (type kwery)))

;; COMPILER ================================================

(def #^{:doc "A map of functions to their type."} where-clause-type
  (atom {"and" ::Recursive
         "or"  ::Recursive
         "not" ::Recursive}))

(defmulti build-env
  "Build environment vector. Replace extracted values with ?."
  {:arglists '([form env])}
  (fn build-env-dispatch [form _]
    (when-let [form (seq form)]
      (get @where-clause-type (-> form first ->string) ::NonRecursive))))

(defmethod build-env ::Recursive
  build-env-recursive
  [[function & forms] env]
  (let [[forms env]
        (reduce
          (fn [[forms env] form]
            (let [[form env] (build-env form env)]
              [(conj forms form) env]))
          [[] env] forms)]
    [(vec (cons function forms)) env]))

(defmethod build-env ::NonRecursive
  build-env-non-recursive
  [[function & args] env]
  (let [[args env]
        (reduce (fn [[args env] arg]
                  (cond
                    (nil? arg)       [(conj args "NULL") env]
                    (self-eval? arg) [(conj args "?")    (conj env arg)]
                    :else            [(conj args arg)    env]))
                [[] env] args)]
    [(vec (cons function args)) env]))

(defmethod build-env nil
  build-env-nil
  [_ _]
  nil)

; Queries

(defn raw
  "Executes a raw SQL statement - This should not be necessary and its
   definately not recommended. If you find ClojureQL lacking in features
   please leave us a note on http://clojureql.lighthouseapp.com"
  [txt]
  (make-type (struct-map sql-raw-statement
                        :statement txt)
            ::Raw))

(defn query*
  "Driver for the query macro. Don't call directly!"
  [table-spec col-spec pred-spec]
  (let [col-spec   (->vector col-spec)
        col-spec   (mapcat fix-prefix col-spec)
        col-spec   (map ->vector col-spec)
        [col-spec col-aliases]     (reduce check-alias [[] {}] col-spec)

        table-spec (->vector table-spec)
        table-spec (map ->vector table-spec)
        [table-spec table-aliases] (reduce check-alias [[] {}] table-spec)

        [pred-spec env]            (build-env pred-spec [])]
    (make-type (struct-map sql-query
                           :columns        col-spec
                           :tables         table-spec
                           :predicates     pred-spec
                           :column-aliases col-aliases
                           :table-aliases  table-aliases
                           :env            env)
               ::Select)))

(defmacro query
  "Define a SELECT query."
  ([table-spec col-spec]
   `(query ~table-spec ~col-spec nil))
  ([table-spec col-spec pred-spec]
   `(query* ~@(map quasiquote* [table-spec col-spec pred-spec]))))

(defn join*
  "Turn a query into JOIN."
  [kwery join-type on-columns]
  (let [join-types {:inner ::InnerJoin
                    :left  ::LeftJoin
                    :right ::RightJoin
                    :full  ::FullJoin}]
    (make-type (struct-map sql-join
                           :on    (list* '= on-columns)
                           :query kwery
                           :env   (kwery :env))
               (get join-types join-type ::InnerJoin))))

(defmacro join
  "Turn a query into a JOIN."
  [kwery join-type on-columns]
  `(join* ~kwery ~join-type ~(quasiquote* on-columns)))

(defn order-by*
  "Driver for the order-by macro. Don't call directly."
  [kwery & columns]
  (when-let [offender (is-and-not? kwery ::Select ::OrderedSelect)]
    (throw (Exception. (str "Unexpected query type: " offender))))
  (let [order   (first columns)
        columns (vec (if (keyword? order) (drop 1 columns) columns))
        order   (if (keyword? order) order :ascending)]
    (make-type (struct-map sql-ordered-query
                           :query   kwery
                           :order   order
                           :columns columns
                           :env     (kwery :env))
               ::OrderedSelect)))

(defmacro order-by
  "Modify the given query to be order according to the given columns. The first
  argument may be one of the keywords :ascending or :descending to choose the
  order used."
  [kwery & columns]
  `(order-by* ~kwery ~@(map quasiquote* columns)))

(defn group-by*
  [kwery & columns]
  (when-let [offender (is-and-not? kwery ::Select ::GroupedSelect)]
    (throw (Exception. (str "Unexpected query type: " offender))))
  (let [columns (vec columns)]
    (make-type (struct-map sql-grouped-query
                           :query   kwery
                           :columns columns
                           :env     (kwery :env))
               ::GroupedSelect)))

(defmacro group-by
  "Modify the given query to be group by the given columns."
  [kwery & columns]
  `(group-by* ~kwery ~@(map quasiquote* columns)))

(defn having*
  "Driver for the having macro. Should not be called directly."
  [kwery pred-spec]
  (when-let [offender (is-and-not? kwery ::Select ::HavingSelect)]
    (throw (Exception. (str "Unexpected query type: " offender))))
  (let [[pred-spec env] (build-env pred-spec (kwery :env))]
    (make-type (struct-map sql-having-query
                           :query      kwery
                           :predicates pred-spec
                           :env        env)
               ::HavingSelect)))

(defmacro having
  "Add a HAVING clause to the given query."
  [kwery pred-spec]
  `(having* ~kwery ~(quasiquote* pred-spec)))

(defn distinct!
  "Modify the given query to return only distinct results."
  [kwery]
  (when-let [offender (is-and-not? kwery ::Select ::DistinctSelect)]
    (throw (Exception. (str "Unexpected query type: " offender))))
  (make-type (struct-map sql-distinct-query
                         :query kwery
                         :env   (kwery :env))
             ::DistinctSelect))

(defn union
  "Build the union of the given queries. The first argument may be the keyword
  :all in order to include all results in the union. Without :all only distinct
  results are included."
  [& kweries]
  (condp = (count kweries)
    0 nil
    1 (first kweries)
    (let [all     (= (first kweries) :all)
          kweries (vec (if all (drop 1 kweries) kweries))]
      (make-type (struct-map sql-union
                             :all     all
                             :queries kweries
                             :env     (vec (mapcat :env kweries)))
                 ::Union))))

(defn intersect
  "Build the intersection of the given queries."
  [& kweries]
  (condp = (count kweries)
    0 nil
    1 (first kweries)
    (make-type (struct-map sql-intersect
                           :queries kweries
                           :env     (vec (mapcat :env kweries)))
               ::Intersect)))

(defn difference
  "Build the difference of the given queries."
  [& kweries]
  (condp = (count kweries)
    0 nil
    1 (first kweries)
    (make-type (struct-map sql-difference
                           :queries kweries
                           :env     (vec (mapcat :env kweries)))
               ::Difference)))

(declare execute-sql)

(defn let-query*
  [thunk]
  (make-type (struct-map sql-let-query :fn thunk) ::LetQuery))

(defmacro let-query
  "Takes a let-style binding vector and returns a new query, which,
  when executed, assigns the queries results to the named locals and
  executes the body. The result of the body is returned as the query's
  result."
  [bindings & body]
  (let [conn    (gensym "let_query_conn__")
        locals  (take-nth 2 bindings)
        kweries (take-nth 2 (rest bindings))]
    `(let-query*
       (fn [~conn]
         (let ~(vec (interleave
                      locals
                      (map (fn [kwery]
                             `(execute-sql ~kwery ~conn))
                           kweries)))
           ~@body)))))

; Data Handling

(defn insert-into*
  "Driver for the insert-into macro. Don't use directly."
  [table col-val-pairs-or-query]
  (let [kwery (when (isa? select-hierarchy
                          (type col-val-pairs-or-query)
                          ::Select)
                col-val-pairs-or-query)
        pairs (when (or (vector? col-val-pairs-or-query)
                        (seq? col-val-pairs-or-query))
                col-val-pairs-or-query)]
    (cond
      kwery
      (make-type (struct-map sql-insert-query
                             :table table
                             :query kwery)
                 ::InsertQuery)

      (and pairs (even? (count pairs)))
      (let [columns (take-nth 2 pairs)
            values  (take-nth 2 (next pairs))]
        (make-type (struct-map sql-insert-values
                               :table   table
                               :columns columns
                               :env     values)
                   ::InsertValues))

      :else
      (throw (Exception. "column/value pairs not balanced or not a query")))))

(defmacro insert-into
  "Insert data into a table."
  [table col-val-pairs-or-query]
  `(insert-into* ~@(map quasiquote* [table col-val-pairs-or-query])))

(defn update*
  "Driver for the update macro. Don't call directly."
  [table col-val-pairs pred-spec]
  (if (even? (count col-val-pairs))
    (let [columns         (vec (take-nth 2 col-val-pairs))
          values          (vec (take-nth 2 (rest col-val-pairs)))
          [pred-spec env] (build-env pred-spec values)]
      (make-type (struct-map sql-update
                             :table      table
                             :columns    columns
                             :predicates pred-spec
                             :env        env)
                 ::Update))
    (throw (Exception. "column/value pairs not balanced"))))

(defmacro update
  "Update the given columns of the given table with the associated values
  where the predicates are satisfied. The relation between columns and
  values is given as a let-style binding vector."
  [table col-val-pairs pred-spec]
  `(update* ~@(map quasiquote* [table col-val-pairs pred-spec])))

(defn delete-from*
  "Driver for the delete-from macro. Don't call directly."
  ([table]
   (delete-from* table nil))
  ([table pred-spec]
   (let [[pred-spec env] (build-env pred-spec [])]
     (make-type (struct-map sql-delete
                            :table      table
                            :predicates pred-spec
                            :env        env)
                ::Delete))))

(defmacro delete-from
  "Delete the entries matching the given predicates from the given table."
  ([table]
   `(delete-from* ~(quasiquote* table)))
  ([table pred-spec]
   `(delete-from* ~@(map quasiquote* [table pred-spec]))))

; Table Handling

(defn alter-table* [table action & options]
  (make-type (struct-map sql-alter-table
               :action  (condp = action
                          'add    ::Add
                          'drop   ::Drop
                          'modify ::Modify
                          'rename ::Rename)
               :options (->arg-lists options)
               :table   table)
    ::AlterTable))

(defmacro alter-table
  [table & options]
  `(alter-table* ~@(map quasiquote* (list* table options))))

(declare batch-statements)

(defn- all-non-nulls
  "Change :non-nulls option if wildcard used (*) to all columns."
  [columns options]
  (if (= '* (:non-nulls options))
    (assoc options :non-nulls (vec (keys columns)))
    options))

(defn create-table*
  "Driver function for create-table macro. Don't use directly."
  [table columns & options]
  (let [columns (apply array-map (->vector columns))
        options (merge {:foreign-key []
                        :primary-key []
                        :non-nulls   []
                        :auto-inc    []
                        :uniques     []
                        :defaults    []
                        :checks      []}
                       (all-non-nulls columns
                                      (apply hash-map options)))]
    (make-type (struct-map sql-create-table
                           :table   table
                           :columns columns
                           :options options)
               ::CreateTable)))

(defmacro create-table
  "Create a table of the given name and the given columns.

   ex.
    (create-table foo [id 'int(11)' name 'varchar(100)' lifestory 'text']
                      :primary-key id :non-nulls id :auto-inc id"
  [table & columns]
  `(create-table* ~@(map quasiquote* (cons table columns))))


(defn create-view*
  "Driver function for create-view macro."
  [view kwery columns]
  (make-type (struct-map sql-create-view
                         :view    view
                         :query   kwery
                         :columns columns)
             ::CreateView))

(defmacro create-view
  "Create a of the given name based on the given query. columns is a
  vector of column names to use in the view."
  [view kwery columns]
  `(create-view* ~(quasiquote* view) ~kwery ~(quasiquote* columns)))

(defn drop-table*
  "Driver function for the drop-table macro. Don't use directly."
  [table & if-exists]
  (let [if-exists (= (first if-exists) :if-exists)]
    (make-type (struct-map sql-drop-table
                           :table     (str table)
                           :if-exists if-exists)
               ::DropTable)))

(defmacro drop-table
  "Drop the given table. Optionally :if-exists might be specified."
  [table & if-exists]
  `(drop-table* ~@(map quasiquote* (cons table if-exists))))

(defn drop-view*
  "Driver function for the drop-view macro. Don't use directly."
  [view & if-exists]
  (let [if-exists (= (first if-exists) :if-exists)]
    (make-type (struct-map sql-drop-view
                           :view      (str view)
                           :if-exists if-exists)
               ::DropView)))

(defmacro drop-view
  "Drop the given view. Optionally :if-exists might be specified."
  [view & if-exists]
  `(drop-view* ~@(map quasiquote* (cons view if-exists))))

(defn batch-statements
  "Execute the given statements in a batch wrapped in a dedicated
  transaction."
  [& statements]
  (make-type (struct-map sql-batch-statement
                         :statements statements)
             ::Batch))
