;; Copyright (c) 2008,2009 Lau B. Jensen <lau.jensen {at} bestinclass.dk
;;                         Meikel Brandmeyer <mb {at} kotka.de>
;; All rights reserved.
;;
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file LICENSE.txt at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by the
;; terms of this license. You must not remove this notice, or any other, from
;; this software.

(ns clojureql.schema
  "Utility functions used to query the database schema.")

(defn- reversed&padded-args [args]
  (concat (reverse args)
          (repeat nil)))

(defn table-like-object
  "Returns a sequence of result sets for the corresponding table-like
  database object."
  [db type args]
  (let [metadata (.getMetaData db)
        [table schema catalog] (reversed&padded-args args)]
    (doall
     (resultset-seq
      (.getTables metadata
                  catalog
                  schema
                  table (into-array [type]))))))

(defn tables
  "Returns all tables found in the given database."
  [db]
  (table-like-object db "TABLE" ["%"]))

(defn table
  "Returns the first matching result set for the corresponding table."
  [db & args]
  (first (table-like-object db "TABLE" args)))

(defn views
  "Returns all views found in the given database."
  [db]
  (table-like-object db "VIEW" ["%"]))

(defn view
  "Returns a sequence of result sets for the corresponding view."
  [db & args]
  (first (table-like-object db "VIEW" args)))

(defn columns
  "Returns a sequence of result sets for the corresponding columns."
  [db & args]
  (let [metadata (.getMetaData db)
        [column table schema catalog]
        (reversed&padded-args args)]
    (doall
     (resultset-seq
      (.getColumns metadata
                   catalog
                   schema
                   table
                   column)))))

(defn data-type
  "Returns the data type for the given column information map."
  [col-info]
  (let [{:keys [type_name char_octet_length]} col-info]
    (if char_octet_length
      (str type_name "(" char_octet_length ")")
      type_name)))
