(ns discljord.logging
  (:require [clojure.java.io :as io]
            [clojure.set :as set])
  (:import (org.apache.log4j Level
                             Logger
                             PropertyConfigurator)))

;; Taken from http://brownsofa.org/blog/2015/06/14/clojure-in-production-logging/
;; All credit goes to Ian Truslove on Jun 14th, 2015

(def log-levels {:all Level/ALL
                 :trace Level/TRACE
                 :debug Level/DEBUG
                 :info Level/INFO
                 :warn Level/WARN
                 :error Level/ERROR
                 :fatal Level/FATAL
                 :off Level/OFF})

(defn- set-logger-level*
  [^org.apache.log4j.Logger logger level]
  {:pre [(log-levels level)]}
  (.setLevel logger (log-levels level)))

(defn set-root-logger-level!
  "Sets the root logger to be at `level`."
  [level]
  (set-logger-level* (Logger/getRootLogger)
                     level))

(defn set-logger-level!
  "Sets the specified `logger` to be at `level`."
  [logger level]
  (set-logger-level* (Logger/getLogger logger)
                     level))

(defn loggers-seq []
  (-> (Logger/getRootLogger)
      .getLoggerRepository
      .getCurrentLoggers
      enumeration-seq))

(defn logger-levels [loggers]
  (into {}
        (for [logger loggers]
          [(.getName logger) ((set/map-invert log-levels) (.getEffectiveLevel logger))])))

(defn set-all-loggers-level!
  "Sets the level of all configured loggers to be at `level`."
  [level]
  (doseq [logger (loggers-seq)]
    (set-logger-level* logger level)))

(defn reload-config!
  "Reconfigures log4j from a log4j.properties file on the classpath"
  []
  (with-open [config-stream
              (-> "log4j.properties"
                  io/resource
                  io/file
                  io/input-stream)]
    (PropertyConfigurator/configure config-stream)))
