;
; Copyright © 2017 Symphony Software Foundation
; SPDX-License-Identifier: Apache-2.0
;
; Licensed under the Apache License, Version 2.0 (the "License");
; you may not use this file except in compliance with the License.
; You may obtain a copy of the License at
;
;     http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.
;

(ns clj-symphony.message
  "Operations related to messages, which combine a human-message (encoded as a subset of HTML called 'MessageMLv2') and a machine readable JSON blob called the 'entity data'."
  (:require [clojure.string      :as s]
            [clojure.java.io     :as io]
            [clj-symphony.user   :as syu]
            [clj-symphony.stream :as sys]))


(defn msgobj->map
  "Converts a SymMessage object into a map."
  [^org.symphonyoss.symphony.clients.model.SymMessage m]
  (if m
    {
      :message-id     (.getId          m)
      :timestamp      (java.util.Date. (Long/valueOf (.getTimestamp m)))
      :stream-id      (.getStreamId    m)
      :user-id        (.getFromUserId  m)
      :type           (.getMessageType m)   ; This seems to be null or blank most of the time...
      :text           (.getMessage     m)
      :attachment     (.getAttachment  m)
      :entity-data    (.getEntityData  m)
    }))


(defn escape
  "Escapes the given string for MessageML."
  [^String m]
  (if m
    (org.apache.commons.lang3.StringEscapeUtils/escapeXml11 m)))


(defn to-plain-text
  "Converts a MessageML message to plain text, converting <p> and <br/> tags into newlines."
  [^String m]
  (let [tmp (org.jsoup.Jsoup/clean m
                                   ""
                                   (.addTags (org.jsoup.safety.Whitelist/none) (into-array String ["br" "p"]))
                                   (.prettyPrint (org.jsoup.nodes.Document$OutputSettings.) true))]
    (org.jsoup.Jsoup/clean tmp
                           ""
                           (org.jsoup.safety.Whitelist/none)
                           (.prettyPrint (org.jsoup.nodes.Document$OutputSettings.) false))))


(defn send-message!
   "Sends the given message (a String), optionally including entity data (a String containing JSON) and an attachment (something compatible with io/file) to the given stream.

See:
  * https://rest-api.symphony.com/docs/messagemlv2 for details on MessageMLv2's formatting capabilities
  * https://rest-api.symphony.com/docs/objects for details on MessageMLv2's entity data capabilities"
  ([c s m]    (send-message! c s m nil nil))
  ([c s m ed] (send-message! c s m ed nil))
  ([^org.symphonyoss.client.SymphonyClient c s ^String m ^String ed a]
    (let [stream-id (sys/stream-id s)]
      (.sendMessage (.getMessagesClient c)
                    (doto (org.symphonyoss.symphony.pod.model.Stream.)
                      (.setId stream-id))
                    (doto
                      (org.symphonyoss.symphony.clients.model.SymMessage.)
                      (.setStreamId   stream-id)
                      (.setMessage    m)
                      (.setEntityData ed)
                      (.setAttachment (io/file a)))))
    nil))


(defn register-listener
  "Registers f, a function with 1 parameter, as a message listener (callback), and returns a handle to that listener so that it can be deregistered later on, if needed.  Listeners registered in this manner are not scoped to any particular stream - they will be sent all messages from all streams that the authenticated connection user is a participant in.

The argument passed to f is a map generated by msgobj->map (see that fn for details).

The value returned by f (if any) is ignored."
  [^org.symphonyoss.client.SymphonyClient c f]
  (let [listener (reify
                   org.symphonyoss.client.services.MessageListener
                   (onMessage [this msg]
                     (f (msgobj->map msg))))]
    (.addMessageListener (.getMessageService c) listener)
    listener))


(defn deregister-listener
  "Deregisters a previously-registered message listener.  Once deregistered, a listener should be discarded (they cannot be reused). Returns true if a valid message listener was deregistered, false otherwise."
  [^org.symphonyoss.client.SymphonyClient c ^org.symphonyoss.client.services.MessageListener l]
  (.removeMessageListener (.getMessageService c) l))
