(ns clj-pnm.core
  "'clj-pnm' core functions namespace."
  (:require [clojure.string :as st]
            #?(:clj [clojure.java.io :as io]
               :cljs [cljs.reader :as r]))
  #?(:clj (:import (java.io Writer))))

;; Utility

(defn comment-line?
  "Returns true if a string is considered a Netpbm format comment line (starts with '#' character).
  Otherwise returns false."
  [line]
  (-> line
      st/trim
      (st/starts-with? "#")))

(defn has-comment-segment?
  "Returns true if the string line contains '#' character i.e.
  beginning of the comment segment."
  [line]
  (some? (st/index-of line "#")))

(defn remove-comment-segment
  "Removes the comment segment from a string line if it exists,
  otherwise return the string unchanged."
  [line]
  (if-let [hash-index (st/index-of line "#")]
    (st/trim (subs line 0 hash-index))
    line))

(defn assoc-comments
  "Associates the comments list to the pnm map if it is present and not empty
  with the :comments key, otherwise returns the map."
  [pnm-map comments]
  (if-not (or (nil? comments) (empty? comments))
    (assoc pnm-map :comments comments)
    pnm-map))

(defn split-on-space-char
  "Splits the given string on space character returning a vector of the parts."
  [s]
  (st/split s #"\s"))

(defn third
  "Returns the third element of the collection."
  [v]
  (nth v 2))

(defn fourth
  "Returns the fourth element of the collection."
  [v]
  (nth v 3))

(defn blank?
  "Returns true if the string is either nil, empty or contains only whitespace."
  [s]
  (or (empty? s) (st/blank? s)))


;; Reading

(defn read-str
  "Reads the object from the string. In Clojure environment uses 'read-string'
  and in ClojureScript environment uses 'cljs.reader/read-string'."
  [s]
  #?(:clj (read-string s)
     :cljs (r/read-string s)))

(defn normalize-whitespace
  "Goes through the collection of strings and reduces multiple white-spaces
  in a string to just one and trims it."
  [v]
  (mapv (fn [s] (st/trim (st/replace s #"\s+" " "))) v))

(defn tokenize
  "Tokenizes a collection of strings by splitting each one on space character
  and flattening the resulting collection."
  [v]
  (flatten (map split-on-space-char v)))

(defn remove-comments
  "Removes all the Netpbm comment lines from the collection of strings."
  [v]
  (remove comment-line? v))

(defn get-comments
  "Returns all the Netpbm comments lines from the collection of strings."
  [v]
  (map
   (comp st/trim #(st/replace % "#" ""))
   (filter has-comment-segment? v)))

(defn remove-comment-segments
  "Removes the comment segments from every string line in the collection."
  [v]
  (mapv remove-comment-segment v))

(defmulti parse
  "Parses the vector of Netpbm lines to a map. Supports 'pbm', 'pgm'
  and 'ppm' formats. Throws an Exception or JS error if format is unsupported."
  (fn [v] (-> v
              first
              st/lower-case
              keyword)))

(defmethod parse :p1
  [v]
  (let [w (read-str (second v))
        h (read-str (third v))
        m (loop [i h
                 r []
                 s (drop 3 v)]
            (if (zero? i)
              r
              (recur (dec i)
                     (conj r (mapv read-str (take w s)))
                     (drop w s))))]
    {:type   :p1
     :width  w
     :height h
     :map    m}))

(defmethod parse :p2
  [v]
  (let [w (read-str (second v))
        h (read-str (third v))
        mv (read-str (fourth v))
        m (loop [i h
                 r []
                 s (drop 4 v)]
            (if (zero? i)
              r
              (recur (dec i)
                     (conj r (mapv read-str (take w s)))
                     (drop w s))))]
    {:type      :p2
     :width     w
     :height    h
     :max-value mv
     :map       m}))

(defmethod parse :p3
  [v]
  (let [w (read-str (second v))
        h (read-str (third v))
        mv (read-str (fourth v))
        m (loop [i h
                 r []
                 s (drop 4 v)]
            (if (zero? i)
              r
              (recur (dec i)
                     (conj r (into [] (partition 3 (map read-str (take (* w 3) s)))))
                     (drop (* w 3) s))))]
    {:type      :p3
     :width     w
     :height    h
     :max-value mv
     :map       m}))

(defmethod parse :default
  [v]
  (let [msg (str "Unsupported file format " (first v) "!")]
    #?(:clj (throw (Exception. msg))
       :cljs (throw (js/Error. msg)))))

(defmulti read-lines
  "Reads string lines for the given source. In Clojure environments the
  function supports string and file sources and in ClojureScript environment
  only string sources."
  (fn [s] (type s)))

(defmethod read-lines :default
  [s]
  (into [] (remove blank? (st/split-lines s))))

#?(:clj
   (defmethod read-lines java.io.File
     [f]
     (with-open [reader (io/reader f)]
       (into [] (remove blank? (line-seq reader))))))

(defn read-pnm
  "Main function for reading Netpbm format. Accepts the string or a file
  and returns a Netpbm map representation."
  [pnm]
  (let [pnm-lines (read-lines pnm)
        comments (get-comments pnm-lines)]
    (-> pnm-lines
        remove-comments
        remove-comment-segments
        normalize-whitespace
        tokenize
        parse
        (assoc-comments comments))))


;; Writing

#?(:clj
   (do
     (defn writeln
       "Writes a string to a Writer. Appends a newline character to a string first.
       The function is exposed only in Clojure environment."
       [^Writer writer line]
       (when line
         (.write writer (str line "\n"))))

     (defn write-comments
       "Writes a comment line to a Writer. Prepends the '# ' sequence to the string.
       The function is exposed only in Clojure environment."
       [^Writer writer comments]
       (when comments
         (doseq [c comments]
           (writeln writer (str "# " c)))))

     (defn write-map
       "Writes the Netpbm bit/value map to a Writer. The function is exposed only in Clojure environment."
       [^Writer writer m]
       (when m
         (doseq [l m]
           (writeln writer (st/join " " (flatten l))))))

     (defn write-pnm
       "The main function for writing Netpbm format to a file. Accepts a Netpbm map representation and a
       file name. The function is exposed only in Clojure environment."
       [pnm file-name]
       (let [f (io/file file-name)]
         (with-open [writer (io/writer f :append true)]
           (doto writer
             (writeln (some-> pnm :type name st/upper-case))
             (writeln (-> pnm :width))
             (writeln (-> pnm :height))
             (writeln (-> pnm :max-value))
             (write-comments (-> pnm :comments))
             (write-map (-> pnm :map)))
           f)))))
