(ns toxi.geom.circle
  (:require
    [toxi.geom.utils :as utils]
    [toxi.math.core :as math])
  (:use
    [toxi.geom.core]
    [toxi.geom.vec2d]
    [toxi.geom.ellipse])
  (:import
    [toxi.geom.types Circle]))

(def ^{:dynamic true} *circle-res* 20)

(defn circle-from-points2
  {:added "0.1"
   :java-id "toxi.geom.Circle.from2Points"}
  [p1 p2]
  (let [m (interpolate p1 p2 0.5)]
    (circle m (distance m p1))))

(defn- circle-intersect
  {:added "0.1"
   :java-id "toxi.geom.Circle.intersectsCircle"}
  [c1 c2]
  (let [delta (sub c2 c1)
        d (mag delta)
        r1 (:radius c1)
        r2 (:radius c2)]
    (if (and (<= d (+ r1 r2)) (>= d (math/abs (- r1 r2))))
      (let [a (/ (+ (- (* r1 r1) (* r2 r2)) (* d d)) (* 2 d))
            invd (/ 1.0 d)
            p (add (utils/swizzle c1 :xy) (scale-n delta (* a invd)))
            h (Math/sqrt (- (* r1 r1) (* a a)))
            perp (scale-n (perpendicular delta) (* h invd))]
        [(add p perp) (sub p perp)]))))

(defn- circle-tangent-points
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.getTangentPoints"}
  [c p]
  (let [m (interpolate c p 0.5)]
    (circle-intersect c (circle m (distance m p)))))

(defn- circle-area
  {:added "0.1"
   :java-id "toxi.geom.Ellipse.getArea"}
  [c] (let[r (:radius c)] (* math/PI r r)))

(defn- circle-bounds
  [c] (let [r (vec2d (:radius c))] (rect (sub c r) (add c r))))

(defn- circle-circumference
  {:added "0.1"
   :java-id "toxi.geom.Circle.getCircumference"}
  [c] (* math/TWO_PI (:radius c)))

(defn- circle-contains-point?
  [c p]
  (<= (distance c p) (:radius c)))

(defn- circle-random-point
  [c]
  (add c (vec2d-from-theta (math/random math/TWO_PI) (math/random (:radius c)))))

(defn- circle->polygon2d
  ([c] (circle->polygon2d c *circle-res*))
  ([c res] (->polygon2d (ellipse c (:radius c)) res)))

(defn- circle-edges
  ([e] (edges (circle->polygon2d e *circle-res*)))
  ([e res] (edges (circle->polygon2d e res))))

(defn extend-circle
  [type]
  (extend-vec2d type)
  (extend type
    IIntersectable {
      :intersect circle-intersect
    }
    IShape {
      :bounds circle-bounds
      :contains-point? circle-contains-point?
      :random-point circle-random-point
    }
    IShape2D {
      :area circle-area
      :bounding-circle identity
      :circumference circle-circumference
      :edges circle-edges
      :->polygon2d circle->polygon2d
    }
    ICircle {
      :tangent-points circle-tangent-points
    }))

(extend-circle Circle)
