(ns toxi.geom.core
  (:require
    [toxi.data.index :as index])
  (:use
    [toxi.core]
    [toxi.math.core]
    [toxi.geom.types])
  (:import
    [toxi.geom.types
     Vec2D Vec3D Vec4D
     Circle Ellipse
     Line2D LineStrip2D Triangle2D Rect Polygon2D
     Matrix4x4 Quaternion
     AABB Sphere
     TriangleMesh3D Face3D]))

(defprotocol IRotatable
  (rotate [e theta])
  (rotate-around-axis [e axis theta])
  (rotate-x [e theta])
  (rotate-y [e theta])
  (rotate-z [e theta])
)

(defprotocol IScalable
  (scale [e v] [e x y] [e x y z])
  (scale-n [e s])
)

(defprotocol ITranslatable
  (translate [e v] [e x y] [e x y z])
)

(defprotocol ITransformable
  (transform [c matrix])
)

(defprotocol IInvertible
  (invert [e])
)

(defprotocol IInterpolatable
  (interpolate [a b t] [a b t f])
)

(defprotocol IIntersectable
  (intersect [a b])
)

(defprotocol IClassifiable
  (classify-point [c p])
)

(defprotocol IVec
  (absv [v])
  (add [a b] [a b c] [a b c & more])
  (angle-between [a b])
  (angle-between-norm [a b])
  (cross [a b])
  (deltav= [a b] [a b delta])
  (dot [a b])
  (limit [v len])
  (mag [v])
  (mag-squared [v])
  (maxv [a b] [a b c])
  (minv [a b] [a b c])
  (normalize [v] [v len])
  (reflect [v r])
  (sub [a b] [a b & more])
  (->cartesian [v])
  (->vec [v])
  (zerov? [v])
)

(defprotocol IVec2D
  (bisect [a b])
  (heading [v])
  (perpendicular [v])
  (slope [v])
  (->3dxy [v])
  (->3dxz [v])
  (->3dyz [v])
  (->polar [v])
)

(defprotocol IVec3D
  (->2dxy [v])
  (->2dxz [v])
  (->2dyz [v])
)

(defprotocol IShape
  (bounds [s])
  (center [s] [s o])
  (centroid [s])
  (contains-point? [s p])
  (random-point [s])
)

(defprotocol IShape2D
  (area [s])
  (bounding-circle [s])
  (circumference [s])
  (edges [s] [s res])
  (point-at [s t])
  (->polygon2d [s] [s res])
)

(defprotocol IShape3D
  (bounding-sphere [s])
  (->mesh [s] [s res])
)

(defprotocol IProximity
  (closest-point [a p])
  (distance [a p])
  (distance-squared [a p])
)

(defprotocol ILine
  (line-direction [l])
  (line-has-endpoint? [l p])
  (line-length [l])
  (line-length-squared [l])
  (line-normal [l])
  (offset-and-grow-by [line offset] [line offset scale] [line offset scale refp])
  (split-line [line len] [line len coll] [line len coll addfirst?])
)

(defprotocol ILine2D
  (theta [l])
)

(defprotocol ITriangle
  (adjust-size [t offset] [t offa offb offc])
  (clockwise? [t])
  (from-barycentric [t b])
  (intersect-triangles [a b])
  (triangle-closest-point-to [t p])
)

(defprotocol IPolygon2D
  (add-vertex [poly v])
  (contains-polygon? [a b])
  (flip [poly])
  (intersect-polygons [a b])
  (vertex-count [poly])
)

(defprotocol IRect
  (bottom[r])
  (bottom-left[r])
  (bottom-right[r])
  (dimensions[r])
  (left[r])
  (right[r])
  (top[r])
  (top-left[r])
  (top-right[r])
)

(defprotocol ICircle
  (tangent-points [c p])
)

(defprotocol IEllipse
  (foci [e])
)

(defprotocol IMatrix
  (matrix-multiply [m m2])
  (matrix-transpose [m])
  (transform-point [m v])
  (transform-normal [m v])  
)

;(defprotocol IMatrix4x4)

(defprotocol IAABB
  (bounds-min [b])
  (bounds-max [b])
)

(defprotocol IMesh
  (add-faces [mesh a b c] [mesh a b c uvcoords] [mesh faces])
  (add-mesh [src target])
  (compile-mesh [mesh] [mesh attribs] [mesh attribs accessors])
  (face-vertices [mesh f])
  (face-attribs [mesh f])
  (transform-mesh [mesh matrix])
  (vertex-at-point [mesh p])
)

(defprotocol IMesh3D
  (compute-face-normals [m])
  (compute-vertex-normals [m])
)


(defn vec2d
  "Creates a new Vec2D instance. Coordinates are cast to Double.
  If the single argument is a collection, the first two items are used as x & y.
  If the single argument is a number, it's used for both XY: (vec2d 1) = (vec2d 1 1).
  If no arguments are specified xy are set to 0.0."
  {:added "0.1"
   :java-id "toxi.geom.Vec2D"}
  ([] (Vec2D. 0.0 0.0))
  ([v]
    (cond
      (map? v) (Vec2D. (:x v) (:y v))
      (coll? v) (Vec2D. (first v) (second v))
      (number? v) (Vec2D. v v)))
  ([^double x ^double y] (Vec2D. (double x) (double y))))

(defn vec3d
  "Creates a new Vec3D instance. Coordinates are cast to Double.
If the single argument is a collection, the first three items are used as x, y & z.
If the single argument is a number, it's used for xyz: (vec3d 1) = (vec3d 1 1 1).
If no arguments are specified, xyz are set to 0.0."
  {:added "0.1"
   :java-id "toxi.geom.Vec3D"}
  ([] (Vec3D. 0.0 0.0 0.0))
  ([v]
    (cond
      (map? v) (Vec3D. (:x v) (:y v) (:z v))
      (coll? v) (Vec3D. (first v) (second v) (nth v 2))
      (number? v) (Vec3D. v v v)))
  ([^double x ^double y ^double z] (Vec3D. (double x) (double y) (double z))))

(defn vec4d
  "Creates a new Vec4D instance. Coordinates are cast to Double.
If argument is a collection, the first four items are used as x, y, z & w.
If no arguments are specified xyz are set to 0.0 and w to 1.0."
  {:added "0.1"
   :java-id "toxi.geom.Vec4D"}
  ([] (Vec4D. 0.0 0.0 0.0 1.0))
  ([coll] (Vec4D. (first coll) (second coll) (nth coll 2) (nth coll 3)))
  ([xyz ^double w] (Vec4D. (:x xyz) (:y xyz) (:z xyz) w))
  ([^double x ^double y ^double z ^double w]
    (Vec4D. (double x) (double y) (double z) (double w))))

(defn line2d
  {:added "0.1"
   :java-id "toxi.geom.Line2D"}
  ([a b] (Line2D. a b))
  ([^double x1 ^double y1 ^double x2 ^double y2] (line2d (vec2d x1 y1) (vec2d x2 y2))))

(defn linestrip2d
  ([] (LineStrip2D. []))
  ([coll] (LineStrip2D. (vec (map vec2d coll))))
  ([v & more] (LineStrip2D. (vec (map vec2d (concat [v] more))))))

(defn triangle2d
  {:added "0.1"
   :java-id "toxi.geom.Triangle2D"}
  ([a b c] (Triangle2D. a b c))
  ([x1 y1 x2 y2 x3 y3]
    (triangle2d (vec2d x1 y1) (vec2d x2 y2) (vec2d x3 y3))))

(defn rect
  {:added "0.1"
   :java-id "toxi.geom.Rect"}
  ([] (Rect. 0 0 0 0))
  ([^Vec2D min ^Vec2D max]
    (let [w (- (:x max) (:x min)) h (- (:y max) (:y min))]
      (Rect. (:x min) (:y min) w h)))
  ([^double x ^double y ^double w ^double h] (Rect. x y w h)))

(defn polygon2d
  ([] (Polygon2D. []))
  ([coll] (Polygon2D. (vec (map vec2d coll))))
  ([v & more] (Polygon2D. (vec (map vec2d (concat [v] more))))))

(defn circle
  {:added "0.1"
   :java-id "toxi.geom.Circle"}
  ([] (Circle. 0.0 0.0 1.0))
  ([radius] (Circle. 0 0 (abs radius)))
  ([origin ^double radius] (Circle. (:x origin) (:y origin) (abs radius)))
  ([^double x ^double y ^double radius] (Circle. x y (abs radius))))

(defn ellipse
  {:added "0.1"
   :java-id "toxi.geom.Ellipse"}
  ([] (Ellipse. 0.0 0.0 (vec2d 1.0)))
  ([origin radii]
    (cond
      (number? radii)
      (Ellipse. (:x origin) (:y origin) (vec2d (abs radii)))
      (map? radii)
      (Ellipse. (:x origin) (:y origin) (vec2d (abs (:x radii)) (abs (:y radii))))))
  ([origin ^double rx ^double ry]
    (Ellipse. (:x origin) (:y origin) (vec2d (abs rx) (abs ry))))
  ([^double x ^double y ^double rx ^double ry]
    (Ellipse. x y (vec2d (abs rx) (abs ry)))))

(defn triangle-mesh3d
  ([] (TriangleMesh3D.
        (index/make-index) ; vertex index
        (index/make-index) ; normal index
        [] ; vertex normal IDs
        {} ; attrib index
        []))) ; face list

(defn face3d
  ([^Integer a ^Integer b ^Integer c]
    (Face3D. (int a) (int b) (int c) (int -1) {}))
  ([^Integer a ^Integer b ^Integer c attribs]
    (Face3D. (int a) (int b) (int c) (int -1) attribs)))

(defn sphere
  ([^double radius]
    (Sphere. 0 0 0 radius))
  ([v ^double radius]
    (Sphere. (:x v) (:y v) (:z v) radius))
  ([x y z radius]
    (Sphere. x y z radius)))

(defn matrix4x4
  {:added "0.1"
   :java-id "toxi.geom.Matrix4x4"}
  ([]
    (Matrix4x4.
      1 0 0 0
      0 1 0 0
      0 0 1 0
      0 0 0 1))
  ([mat]
    (Matrix4x4.
      (get mat 0) (get mat 1) (get mat 2) (get mat 3)
      (get mat 4) (get mat 5) (get mat 6) (get mat 7)
      (get mat 8) (get mat 9) (get mat 10) (get mat 11)
      (get mat 12) (get mat 13) (get mat 14) (get mat 15)))
  ([m00 m01 m02 m03 m10 m11 m12 m13 m20 m21 m22 m23 m30 m31 m32 m33]
    (Matrix4x4.
      m00 m01 m02 m03
      m10 m11 m12 m13
      m20 m21 m22 m23
      m30 m31 m32 m33)))

(defn aabb
  {:added "0.1"
   :java-id "toxi.geom.AABB"}
  ([] (AABB. 0 0 0 (vec3d 0.5 0.5 0.5)))
  ([extent] (AABB. 0 0 0 (vec3d extent)))
  ([origin extent] (AABB. (:x origin) (:y origin) (:z origin) extent))
  ([x y z ex ey ez] (AABB. x y z (vec3d ex ey ez))))

(defn quaternion
  {:added "0.1"
   :java-id "toxi.geom.Quaternion"}
  ([] (Vec4D. 0.0 0.0 0.0 1.0))
  ([^double w xyz] (Vec4D. (:x xyz) (:y xyz) (:z xyz) w))
  ([^double w ^double x ^double y ^double z] (Vec4D. x y z w)))
