(ns toxi.geom.linestrip2d
  (:require
    [toxi.math.core :as math]
    [toxi.collections :as coll :only [successive-pairs]])
  (:use
    [toxi.geom.core]
    [toxi.geom.vec2d]
    [toxi.geom.line2d])
  (:import
    [toxi.geom.types Line2D LineStrip2D]))

(defn- ls2d-length
  [{:keys[vertices]}]
  (if (> (count vertices) 1)
    (reduce
      (fn[len [p q]] (+ len (distance p q)))
      0 (coll/successive-pairs vertices))
    0))

(defn- ls2d-length-squared
  [{:keys[vertices]}]
  (if (> (count vertices) 1)
    (reduce
      (fn[len [p q]] (+ len (distance-squared p q)))
      0 (coll/successive-pairs vertices))
    0))

(defn- ls2d-split
  ([strip ^double len] (ls2d-split strip len [] true))
  ([strip ^double len coll] (ls2d-split strip len coll true))
  ([strip ^double len coll addfirst?]
    (let[vertices (:vertices strip)]
      (if (> (count vertices) 1)
        (reduce
          (fn[acc [p q i]]
            (split-line (line2d p q) len acc (and addfirst? (= i 0))))
          []
          (map (fn[p q i] [p q i]) (butlast vertices) (drop 1 vertices) (range)))))))

(defn- ls2d-point-on-line
  [{:keys[vertices] :as strip} t]
  (if (> (count vertices) 1)
    (loop[total (ls2d-length strip)
          verts (drop 1 vertices)
          p (first vertices)
          q (first verts)
          offp 0
          offq (/ (distance p q) total)]
      (if (and (<= offp t) (>= offq t))
        (interpolate p q (math/map-interval t offp offq 0 1))
        (if-not (nil? (next verts))
          (let[rv (rest verts)
               nq (first rv)
               noffq (+ offq (/ (distance q nq) total))]
            (recur total rv q nq offq noffq)))))))

(defn- ls2d-intersect
  [{:keys[vertices]} line]
  (reduce
    (fn[acc [p q]]
      (conj acc (intersect (line2d p q) line)))
    [] (coll/successive-pairs vertices)))

(defn extend-linestrip2d
  [type]
  (extend type
;    IScalable {
;      :scale-n l2d-scale-length
;    }
;    IShape {
;      :bounds l2d-bounds
;      :centroid l2d-midpoint
;      :contains-point? l2d-contains-point?
;      :random-point l2d-random-point
;    }
    IIntersectable {
      :intersect ls2d-intersect
    }
    IShape2D {
      :area (fn[l] 0)
;      :bounding-circle ls2d-bounding-circle
;      :circumference (fn[l] (* 2 (l2d-length l)))
;      :edges ls2d-edges
;      :->polygon2d (fn[l] (polygon2d [(:a l) (:b l)]))
      :point-at ls2d-point-on-line
    }
    ILine {
;      :line-closest-point-to l2d-closest-point-to
;      :line-direction l2d-direction
;      :line-distance-to-point l2d-distance-to-point
;      :line-has-endpoint? l2d-has-endpoint?
      :line-length ls2d-length
      :line-length-squared ls2d-length-squared
;      :offset-and-grow-by l2d-offset-and-grow-by
      :split-line ls2d-split
    }
    ))

(extend-linestrip2d LineStrip2D)
