(ns toxi.geom.line2d
  (:require
    [toxi.math.core :as math]
    [toxi.geom.core :as geom]
    [toxi.geom.common :as common]
    [toxi.geom.vec2d :as v2d])
  (:import
    [toxi.geom.types Line2D]))

(defn- classify-point
  [line p]
  (let [a (:a line)
        normal (geom/perpendicular (geom/sub (:b line) a))
        d (geom/dot (geom/sub p a) normal)]
    (Math/signum (double d))))

(defn- distance-squared-to-point
  [line p] (geom/distance-squared (common/closest-point-to-line line p) p))

(defn- distance-to-point
  [line p] (Math/sqrt (distance-squared-to-point line p)))

(defn- intersect
  [l1 l2]
  (let [ax1 (:x (:a l1)) ay1 (:y (:a l1)) bx1 (:x (:b l1)) by1 (:y (:b l1))
        ax2 (:x (:a l2)) ay2 (:y (:a l2)) bx2 (:x (:b l2)) by2 (:y (:b l2))
        denom (- (* (- by2 ay2) (- bx1 ax1)) (* (- bx2 ax2) (- by1 ay1)))
        na (- (* (- bx2 ax2) (- ay1 ay2)) (* (- by2 ay2) (- ax1 ax2)))
        nb (- (* (- bx1 ax1) (- ay1 ay2)) (* (- by1 ay1) (- ax1 ax2)))]
    (if-not (zero? denom)
      (let [ua (/ na denom) ub (/ nb denom) ipos (common/point-on-line l1 ua)]
        (if (and (>= ua 0.0) (<= ua 1.0) (>= ub 0.0) (<= ub 1.0))
          {:type :intersect :pos ipos :ua ua :ub ub}
          {:type :no-intersect :pos ipos :ua ua :ub ub}))
      (if (= 0.0 na nb)
        (if (zero? (distance-squared-to-point l1 (:a l2)))
          {:type :coincident}
          {:type :coincident-no-intersect})
        {:type :parallel}))))

(defn- offset-and-grow-by
  ([line offset] (offset-and-grow-by line offset 1.0 nil))
  ([line offset scale] (offset-and-grow-by line offset scale nil))
  ([line offset scale ref]
    (let [m (common/midpoint line)
          d (common/direction line)
          ds (geom/scale-n d scale)
          t (geom/perpendicular d)
          n (geom/normalize t
              (if (and
                    (not (nil? ref))
                    (neg? (geom/dot (geom/sub m ref) t)))
                (* -1 offset) offset))
          a (geom/add (:a line) n)
          b (geom/add (:b line) n)]
      (assoc line :a (geom/sub a ds) :b (geom/add b ds)))))

(defn- bounding-circle
  [line] (geom/circle (common/midpoint line) (* 0.5 (common/length line))))

(defn- transform
  [line mat] (common/transform2d-keys* line [:a :b] mat))

(defn- translate
  ([line t] (common/translate-keys* line [:a :b] t))
  ([line x y] (common/translate-keys* line [:a :b] x y)))

(defn- scale
  ([line s] (common/scale-keys* line [:a :b] s))
  ([line x y] (common/scale-keys* line [:a :b] x y)))

(defn- scale-n
  [line s] (common/scale-n-keys* line [:a :b] s))

(defn- rotate
  [line theta] (common/rotate2d-keys* line [:a :b] theta))

(defn extend-line2d
  [type]
  (extend type
    geom/IClassifiable {
      :classify-point classify-point
    }
    geom/IIntersectable {
      :intersect intersect
    }
    geom/IRotatable {
      :rotate rotate
    }
    geom/IScalable {
      :scale scale
      :scale-n scale-n
    }
    geom/ITranslatable {
      :translate translate
    }
    geom/ITransformable {
      :transform transform
    }
    geom/IProximity {
      :closest-point common/closest-point-to-line
      :distance distance-to-point
      :distance-squared distance-squared-to-point
    }
    geom/IHeading2D {
      :direction common/direction
      :heading (fn[l] (geom/heading (geom/sub (:b l) (:a l))))
      :slope (fn[l] (geom/slope (geom/sub (:b l) (:a l))))
    }
    geom/IShape {
      :bounds (fn[l] (geom/rect (geom/minv (:a l) (:b l)) (geom/maxv (:a l) (:b l))))
      :centroid common/midpoint
      :contains-point? (fn[l p] (zero? (classify-point l p)))
      :random-point (fn[l] (common/point-on-line l (math/random)))
    }
    geom/IShape2D {
      :area (fn[l] 0)
      :bounding-circle bounding-circle
      :edges (fn[l] [l])
      :point-at common/point-on-line
      :->polygon2d (fn[l] (geom/polygon2d [(:a l) (:b l)]))
    }
    geom/IPath {
      :has-endpoint? (fn[l p] (common/has-endpoint? (:a l) (:b l) p))
      :length common/length
      :length-squared common/length-squared
      :->segments common/split-line
      :vertex-count (fn[l] 2)
    }
    geom/ILine {
      :normal (fn[l] (-> (geom/sub (:b l) (:a l)) geom/perpendicular geom/normalize))
      :offset-and-grow-by offset-and-grow-by
      :scale-length common/scale-length
    }
    geom/IFlippable {
      :flip common/line-flip
    }
    ))

(extend-line2d Line2D)
