(ns toxi.geom.linestrip2d
  (:require
    [toxi.math.core :as math]
    [toxi.data.collections :as coll :only [successive-pairs successive-pairs-indexed]]
    [toxi.geom.core :as geom]
    [toxi.geom.common :as common]
    [toxi.geom.vec2d :as v2d]
    [toxi.geom.line2d])
  (:import
    [toxi.geom.types LineStrip2D]))

(defn- length
  [{:keys[vertices]}]
  (if (> (count vertices) 1)
    (reduce
      (fn[len [p q]] (+ len (geom/distance p q)))
      0 (coll/successive-pairs vertices))
    0))

(defn- length-squared
  [{:keys[vertices]}]
  (if (> (count vertices) 1)
    (reduce
      (fn[len [p q]] (+ len (geom/distance-squared p q)))
      0 (coll/successive-pairs vertices))
    0))

(defn- split*
  ([strip ^double len] (split* strip len [] true))
  ([strip ^double len coll] (split* strip len coll true))
  ([strip ^double len coll addfirst?]
    (let[vertices (:vertices strip)]
      (if (> (count vertices) 1)
        (reduce
          (fn[acc [p q i]]
            (geom/->segments (geom/line2d p q) len acc (and addfirst? (zero? i))))
          []
          (coll/successive-pairs-indexed vertices))))))

(defn- point-on-line
  [{:keys[vertices] :as strip} t]
  (if (> (count vertices) 1)
    (cond
      (<= t 0.0) (first vertices)
      (>= t 1.0) (last vertices)
      :else
      (loop[total (length strip)
            verts (drop 1 vertices)
            p (first vertices)
            q (first verts)
            offp 0
            offq (/ (geom/distance p q) total)]
        (if (and (<= offp t) (>= offq t))
          (geom/interpolate p q (math/map-interval t offp offq 0 1))
          (if-not (nil? (next verts))
            (let[rv (rest verts)
                 nq (first rv)
                 noffq (+ offq (/ (geom/distance q nq) total))]
              (recur total rv q nq offq noffq))))))))

(defn- sample-uniform
  ([strip ^double len] (sample-uniform strip len []))
  ([strip ^double len coll]
    (let[total-length (length strip)
         delta (/ len total-length)
         samples (conj (vec (map (partial point-on-line strip) (range 0 1.0 delta)))
                       (last (:vertices strip)))]
      samples)))
  
(defn- intersect
  [{:keys[vertices]} line]
  (reduce
    (fn[acc e]
      (conj acc (geom/intersect e line)))
    [] (common/edges* vertices)))

(defn- contains-point?
  [strip p]
  (loop[edges (common/edges* (:vertices strip))]
    (when-let[e (first edges)]
      (if (zero? (geom/distance e p))
        true
        (recur (rest edges))))))

(defn- closest-point
  [strip p]
  (common/closest-point* (:vertices strip) p))

(defn- distance-squared
  [strip p]
  (geom/distance-squared (closest-point strip p) p))

(defn- distance
  [strip p]
  (geom/distance (closest-point strip p) p))

(defn extend-linestrip2d
  [type]
  (extend type
    geom/IRotatable {
      :rotate common/rotate2d*
    }
    geom/IScalable {
      :scale common/scale*
      :scale-n common/scale-n*
    }
    geom/ITranslatable {
      :translate common/translate*
    }
    geom/ITransformable {
      :transform common/transform2d*
    }
    geom/IIntersectable {
      :intersect intersect
    }
    geom/IProximity {
      :closest-point closest-point
      :distance distance
      :distance-squared distance-squared
    }
    geom/IShape {
      :bounds (fn[strip] (common/bounds* (:vertices strip)))
      :centroid (fn[strip] (common/centroid* (:vertices strip)))
      :contains-point? contains-point?
      :random-point (fn[strip] (point-on-line strip (math/random)))
    }
    geom/IShape2D {
      :area (fn[l] 0)
      :bounding-circle (fn[strip] (common/bounding-circle* (common/centroid* (:vertices strip)) (:vertices strip)))
      :edges (fn[s] (common/edges* (:vertices s)))
      :->polygon2d (fn[s] (geom/polygon2d (:vertices s)))
      :point-at point-on-line
    }
;    geom/ILine {
;      :offset-and-grow-by l2d-offset-and-grow-by
;    }
    geom/IPath {
      :conjv (fn[strip coll] (common/conjv* strip coll))
      :has-endpoint? (fn[strip p] (common/has-endpoint? (:vertices strip) p))
      :length length
      :length-squared length-squared
      :->segments split*
      :sample-uniform sample-uniform
      :vertex-count (fn[strip] (count (:vertices strip)))
    }
    ))

(extend-linestrip2d LineStrip2D)
