(ns toxi.math.matrix4x4
  (:require
    [toxi.geom.utils :as utils])
  (:use
    [toxi.math.core :as math]
    [toxi.geom.core :as geom])
  (:import
    [toxi.math.types Matrix4x4]))

(def M4X4-IDENTITY
  (math/matrix4x4
    1.0 0.0 0.0 0.0
    0.0 1.0 0.0 0.0
    0.0 0.0 1.0 0.0
    0.0 0.0 0.0 1.0))

(defn m4x4-multiply
  {:added "0.1"
   :java-id "toxi.geom.Matrix4x4.multiply"}
  [mat mat2]
  (let [m00 (:m00 mat) m01 (:m01 mat) m02 (:m02 mat) m03 (:m03 mat)
        m10 (:m10 mat) m11 (:m11 mat) m12 (:m12 mat) m13 (:m13 mat)
        m20 (:m20 mat) m21 (:m21 mat) m22 (:m22 mat) m23 (:m23 mat)
        m30 (:m30 mat) m31 (:m31 mat) m32 (:m32 mat) m33 (:m33 mat)
        n00 (:m00 mat2) n01 (:m01 mat2) n02 (:m02 mat2) n03 (:m03 mat2)
        n10 (:m10 mat2) n11 (:m11 mat2) n12 (:m12 mat2) n13 (:m13 mat2)
        n20 (:m20 mat2) n21 (:m21 mat2) n22 (:m22 mat2) n23 (:m23 mat2)
        n30 (:m30 mat2) n31 (:m31 mat2) n32 (:m32 mat2) n33 (:m33 mat2)]
    (math/matrix4x4
      (+ (+ (+ (* m00 n00) (* m01 n10)) (* m02 n20)) (* m03 n30))
      (+ (+ (+ (* m00 n01) (* m01 n11)) (* m02 n21)) (* m03 n31))
      (+ (+ (+ (* m00 n02) (* m01 n12)) (* m02 n22)) (* m03 n32))
      (+ (+ (+ (* m00 n03) (* m01 n13)) (* m02 n23)) (* m03 n33))

      (+ (+ (+ (* m10 n00) (* m11 n10)) (* m12 n20)) (* m13 n30))
      (+ (+ (+ (* m10 n01) (* m11 n11)) (* m12 n21)) (* m13 n31))
      (+ (+ (+ (* m10 n02) (* m11 n12)) (* m12 n22)) (* m13 n32))
      (+ (+ (+ (* m10 n03) (* m11 n13)) (* m12 n23)) (* m13 n33))

      (+ (+ (+ (* m20 n00) (* m21 n10)) (* m22 n20)) (* m23 n30))
      (+ (+ (+ (* m20 n01) (* m21 n11)) (* m22 n21)) (* m23 n31))
      (+ (+ (+ (* m20 n02) (* m21 n12)) (* m22 n22)) (* m23 n32))
      (+ (+ (+ (* m20 n03) (* m21 n13)) (* m22 n23)) (* m23 n33))

      (+ (+ (+ (* m30 n00) (* m31 n10)) (* m32 n20)) (* m33 n30))
      (+ (+ (+ (* m30 n01) (* m31 n11)) (* m32 n21)) (* m33 n31))
      (+ (+ (+ (* m30 n02) (* m31 n12)) (* m32 n22)) (* m33 n32))
      (+ (+ (+ (* m30 n03) (* m31 n13)) (* m32 n23)) (* m33 n33)))))

(defn m4x4-set-position
  {:added "0.1"
   :java-id :none}
  ([mat v]
    (assoc mat :m03 (:x v) :m13 (:y v) :m23 (:z v)))
  ([mat ^double x ^double y ^double z]
    (assoc mat :m03 x :m13 y :m23 z)))

(defn m4x4-set-scale
  {:added "0.1"
   :java-id :none}
  ([mat v] (assoc mat :m00 (:x v) :m11 (:y v) :m22 (:z v)))
  ([mat ^double x ^double y ^double z]
    (assoc mat :m00 x :m11 y :m22 z)))

(defn m4x4-scale
  {:added "0.1"
   :java-id "toxi.geom.Matrix4x4.scale"}
  ([mat v] 
      (m4x4-scale mat (:x v) (:y v) (:z v)))
  ([mat ^double x ^double y ^double z]
    (m4x4-multiply mat (m4x4-set-scale M4X4-IDENTITY x y z))))

(defn- m4x4-scale-n
  [mat n]
  (m4x4-multiply mat (m4x4-set-scale M4X4-IDENTITY n n n)))

(defn m4x4-translate
  {:added "0.1"
   :java-id "toxi.geom.Matrix4x4.translate"}
  ([mat v]
    (m4x4-translate mat (:x v) (:y v) (:z v)))
  ([mat ^double x ^double y ^double z]
    (m4x4-multiply mat (m4x4-set-position M4X4-IDENTITY x y z))))

(defn m4x4-rotate-around-axis
  {:added "0.1"
   :java-id "toxi.geom.Matrix4x4.rotateAroundAxis"}
  [mat axis ^double theta]
  (let [x (:x axis) y (:y axis) z (:z axis)
        s (Math/sin theta) c (Math/cos theta)
        t (- 1.0 c) tx (* t x) ty (* t y)
        rot (math/matrix4x4
              (+ (* tx x) c) (+ (* tx y) (* s z)) (- (* tx z) (* s y)) 0
              (- (* tx y) (* s z)) (+ (* ty y) c) (+ (* ty z) (* s x)) 0
              (+ (* tx z) (* s y)) (- (* ty z) (* s x)) (+ (* t z z) c) 0
              0 0 0 1)]
    (m4x4-multiply mat rot)))

(defn m4x4-rotate-x
  {:added "0.1"
   :java-id "toxi.geom.Matrix4x4.rotate-x"}
  [mat theta]
  (let [sin (Math/sin theta) cos (Math/cos theta)]
    (m4x4-multiply mat (assoc M4X4-IDENTITY :m11 cos :m22 cos :m21 sin :m12 (* -1.0 sin)))))

(defn m4x4-rotate-y
  {:added "0.1"
   :java-id "toxi.geom.Matrix4x4.rotate-y"}
  [mat ^double theta]
  (let [sin (Math/sin theta) cos (Math/cos theta)]
    (m4x4-multiply mat (assoc M4X4-IDENTITY :m00 cos :m22 cos :m02 sin :m20 (* -1.0 sin)))))

(defn m4x4-rotate-z
  {:added "0.1"
   :java-id "toxi.geom.Matrix4x4.rotate-z"}
  [mat ^double theta]
  (let [sin (Math/sin theta) cos (Math/cos theta)]
    (m4x4-multiply mat (assoc M4X4-IDENTITY :m00 cos :m11 cos :m10 sin :m01 (* -1.0 sin)))))

(defn m4x4-transform-point
  "Transforms the vector parameter by the given matrix.
If the vector has no w component, it is assumed to be one."
  {:added "0.1"
   :java-id "toxi.geom.Matrix4x4.applyTo"}
  [mat v]
  (if (nil? (:w v))
    (let [vx (:x v) vy (:y v) vz (:z v)
          x (+ (+ (+ (* vx (:m00 mat)) (* vy (:m01 mat))) (* vz (:m02 mat))) (:m03 mat))
          y (+ (+ (+ (* vx (:m10 mat)) (* vy (:m11 mat))) (* vz (:m12 mat))) (:m13 mat))
          z (+ (+ (+ (* vx (:m20 mat)) (* vy (:m21 mat))) (* vz (:m22 mat))) (:m23 mat))]
      (assoc v :x x :y y :z z))
    (let [vx (:x v) vy (:y v) vz (:z v) vw (:w v)
          x (+ (+ (+ (* vx (:m00 mat)) (* vy (:m01 mat))) (* vz (:m02 mat))) (* vw (:m03 mat)))
          y (+ (+ (+ (* vx (:m10 mat)) (* vy (:m11 mat))) (* vz (:m12 mat))) (* vw (:m13 mat)))
          z (+ (+ (+ (* vx (:m20 mat)) (* vy (:m21 mat))) (* vz (:m22 mat))) (* vw (:m23 mat)))
          w (+ (+ (+ (* vx (:m30 mat)) (* vy (:m31 mat))) (* vz (:m32 mat))) (* vw (:m33 mat)))]
      (assoc v :x x :y y :z z :w w))))

(defn m4x4-transform-normal
  "Transforms the normal parameter by the given matrix.
The fourth element of the normal is *always* assumed to be zero.
This means only scale/rotation (no translation/position) is
taken into account."
  {:added "0.1"
   :java-id :none}
  [mat n]
  (let [nx (:x n) ny (:y n) nz (:z n)
        x (+ (+ (* nx (:m00 mat)) (* ny (:m01 mat))) (* nz (:m02 mat)))
        y (+ (+ (* nx (:m10 mat)) (* ny (:m11 mat))) (* nz (:m12 mat)))
        z (+ (+ (* nx (:m20 mat)) (* ny (:m21 mat))) (* nz (:m22 mat)))]
    (assoc n :x x :y y :z z)))

(defn m4x4-transpose
  "Computes the transpose of the given matrix.
Flips between row-major and column-major ordering."
  {:added "0.1"
   :java-id "toxi.geom.Matrix4x4.transpose"}
  [mat]
  (math/matrix4x4
    (:m00 mat) (:m10 mat) (:m20 mat) (:m30 mat)
    (:m01 mat) (:m11 mat) (:m21 mat) (:m31 mat)
    (:m02 mat) (:m12 mat) (:m22 mat) (:m32 mat)
    (:m03 mat) (:m13 mat) (:m23 mat) (:m33 mat)))

(defn- m4x4-invert
  [mat]
  (let [tmat (m4x4-transpose mat)
        m00 (:m00 tmat) m01 (:m01 tmat) m02 (:m02 tmat) m03 (:m03 tmat)
        m10 (:m10 tmat) m11 (:m11 tmat) m12 (:m12 tmat) m13 (:m13 tmat)
        m20 (:m20 tmat) m21 (:m21 tmat) m22 (:m22 tmat) m23 (:m23 tmat)
        m30 (:m30 tmat) m31 (:m31 tmat) m32 (:m32 tmat) m33 (:m33 tmat)
        t00 (* m22 m33) t01 (* m23 m32) t02 (* m21 m33) t03 (* m23 m31)
        t04 (* m21 m32) t05 (* m22 m31) t06 (* m20 m33) t07 (* m23 m30)
        t08 (* m20 m32) t09 (* m22 m30) t10 (* m20 m31) t11 (* m21 m30)
        q00 (* m02 m13) q01 (* m03 m12) q02 (* m01 m13) q03 (* m03 m11)
        q04 (* m01 m12) q05 (* m02 m11) q06 (* m00 m13) q07 (* m03 m10)
        q08 (* m00 m12) q09 (* m02 m10) q10 (* m00 m11) q11 (* m01 m10)
        s00 m20 s01 m21 s02 m22 s03 m23 s04 m30 s05 m31 s06 m32 s07 m33
        d00 (- (+ (+ (* t00 m11) (* t03 m12)) (* t04 m13))
               (+ (+ (* t01 m11) (* t02 m12)) (* t05 m13)))
        d01 (- (+ (+ (* t01 m10) (* t06 m12)) (* t09 m13))
               (+ (+ (* t00 m10) (* t07 m12)) (* t08 m13)))
        d02 (- (+ (+ (* t02 m10) (* t07 m11)) (* t10 m13))
               (+ (+ (* t03 m10) (* t06 m11)) (* t11 m13)))
        d03 (- (+ (+ (* t05 m10) (* t08 m11)) (* t11 m12))
               (+ (+ (* t04 m10) (* t09 m11)) (* t10 m12)))
        d04 (- (+ (+ (* t01 m01) (* t02 m02)) (* t05 m03))
               (+ (+ (* t00 m01) (* t03 m02)) (* t04 m03)))
        d05 (- (+ (+ (* t00 m00) (* t07 m02)) (* t08 m03))
               (+ (+ (* t01 m00) (* t06 m02)) (* t09 m03)))
        d06 (- (+ (+ (* t03 m00) (* t06 m01)) (* t11 m03))
               (+ (+ (* t02 m00) (* t07 m01)) (* t10 m03)))
        d07 (- (+ (+ (* t04 m00) (* t09 m01)) (* t10 m02))
               (+ (+ (* t05 m00) (* t08 m01)) (* t11 m02)))
        d08 (- (+ (+ (* q00 s05) (* q03 s06)) (* q04 s07))
               (+ (+ (* q01 s05) (* q02 s06)) (* q05 s07)))
        d09 (- (+ (+ (* q01 s04) (* q06 s06)) (* q09 s07))
               (+ (+ (* q00 s04) (* q07 s06)) (* q08 s07)))
        d10 (- (+ (+ (* q02 s04) (* q07 s05)) (* q10 s07))
               (+ (+ (* q03 s04) (* q06 s05)) (* q11 s07)))
        d11 (- (+ (+ (* q05 s04) (* q08 s05)) (* q11 s06))
               (+ (+ (* q04 s04) (* q09 s05)) (* q10 s06)))
        d12 (- (+ (+ (* q02 s02) (* q05 s03)) (* q01 s01))
               (+ (+ (* q04 s03) (* q00 s01)) (* q03 s02)))
        d13 (- (+ (+ (* q08 s03) (* q00 s00)) (* q07 s02))
               (+ (+ (* q06 s02) (* q09 s03)) (* q01 s00)))
        d14 (- (+ (+ (* q06 s01) (* q11 s03)) (* q03 s00))
               (+ (+ (* q10 s03) (* q02 s00)) (* q07 s01)))
        d15 (- (+ (+ (* q10 s02) (* q04 s00)) (* q09 s01))
               (+ (+ (* q08 s01) (* q11 s02)) (* q05 s00)))
        det (/ 1.0
               (+ (+ (+ (* m00 d00) (* m01 d01)) (* m02 d02)) (* m03 d03)))]
    (math/matrix4x4 (vec (map #(* % det)
                    [d00 d01 d02 d03
                     d04 d05 d06 d07
                     d08 d09 d10 d11
                     d12 d13 d14 d15])))))

(defn frustum
  "Sets up a viewing frustum, which is shaped like a truncated pyramid with the
  camera where the point of the pyramid would be.
  This emulates the OpenGL function glFrustum()."
  [l r b t n f]
    (math/matrix4x4
      (/ (* 2.0 n) (- r l)) 0 (/ (+ r l) (- r l)) 0
      0 (/ (* 2.0 n) (- t b)) (/ (+ t b) (- t b)) 0
      0 0 (* -1 (/ (+ f n) (- f n))) (/ (* -2 f n) (- f n))
      0 0 -1 0))
    
(defn ortho
  "Returns an orthographic projection, in which objects are the same size no
  matter how far away or nearby they are.
  This emulates the OpenGL function glOrtho()."
  [l r b t n f]
  (math/matrix4x4
    (/ 2.0 (- r l)) 0 0 (/ (+ r l) (- r l))
    0 (/ 2.0 (- t b)) 0 (/ (+ t b) (- t b))
    0 0 (/ -2.0 (- f n)) (/ (+ f n) (- f n))
    0 0 0 1))

(defn perspective
  "Returns a perspective transform matrix, which makes far away objects appear
  smaller than nearby objects. The `aspect` argument should be the width
  divided by the height of your viewport and `fov` is the vertical angle
  of the field of view in degrees.
  This emulates the OpenGL function gluPerspective()."
  [fov aspect near far]
    (let [y (* (Math/tan (* 0.5 (math/radians fov))) near)
          x (* aspect y)]
      (frustum (* -1 x) x (* -1 y) y near far)))

(defn look-at
  "Returns a matrix that puts the camera at the eye position looking
  toward the target point with the given up direction.
  This emulates the OpenGL function `gluLookAt()`."
  [eye target upvec]
  (let [f (geom/normalize (geom/sub eye target))
        s (geom/normalize (geom/cross upvec f))
        t (geom/normalize (geom/cross f s))]
    (math/matrix4x4
      (:x s) (:y s) (:z s) (* -1 (geom/dot s eye))
      (:x t) (:y t) (:z t) (* -1 (geom/dot t eye))
      (:x f) (:y f) (:z f) (* -1 (geom/dot f eye))
      0 0 0 1)))

(defn extend-matrix4x4
  [type]
  (extend type
    IInvertible {
      :invert m4x4-invert
    }
    IRotatable {
      :rotate-around-axis m4x4-rotate-around-axis
      :rotate-x m4x4-rotate-x
      :rotate-y m4x4-rotate-y
      :rotate-z m4x4-rotate-z
    }
    IScalable {
      :scale m4x4-scale
      :scale-n m4x4-scale-n
    }
    ITranslatable {
      :translate m4x4-translate
    }
    IMatrix {
      :matrix-multiply m4x4-multiply
      :matrix-transpose m4x4-transpose
      :transform-point m4x4-transform-point
      :transform-normal m4x4-transform-normal
    }
  ))

(extend-matrix4x4 Matrix4x4)