(ns toxi.geom.common
  (:require
    [toxi.data.collections :as coll :only [successive-pairs apply-to-keys]]
    [toxi.geom.core :as geom]
    [toxi.geom.utils :as utils]
    [toxi.geom.mesh.core :as mesh]
    [toxi.math.core :as math]
    ))

(defn bounds*
  [vertices]
  (if-not (nil? (seq vertices))
    (let [[tlx tly brx bry]
          (reduce
            (fn[[x1 y1 x2 y2] {:keys [x y]}]
              [(min x1 x) (min y1 y) (max x2 x) (max y2 y)])
            [(Double/MAX_VALUE) (Double/MAX_VALUE)
             (Double/MIN_VALUE) (Double/MIN_VALUE)]
            vertices)]
      (geom/rect tlx tly (- brx tlx) (- bry tly)))))

(defn bounding-circle*
  [centroid vertices]
  (if (and centroid (not (nil? (seq vertices))))
    (geom/circle centroid (Math/sqrt (reduce #(max %1 (geom/distance-squared centroid %2)) 0 vertices)))))

(defn centroid*
  [vertices]
  (if-not (nil? (seq vertices))
    (geom/scale-n (reduce geom/add vertices) (/ 1.0 (count vertices)))))

(defn conjv*
  [type vertices]
  (if (or (vector? vertices) (list? vertices) (seq? vertices))
    (assoc type :vertices (reduce conj (:vertices type) vertices))
    (assoc type :vertices (conj (:vertices type) vertices))))

(defn edges*
  [vertices]
  (if-not (nil? (seq vertices))
    (map (fn[[p q]] (geom/line2d p q)) (coll/successive-pairs vertices))))

(defn apply-transform*
  [type f arg]
  (assoc type :vertices (vec (map (fn[v] (f v arg)) (:vertices type)))))

; TODO use matrix3x3 instead of 4x4
(defn transform2d*
  [type matrix]
  (assoc type :vertices
         (vec
           (map #(->> (geom/->3dxy %) (math/transform-point matrix) geom/->2dxy)
                (:vertices type)))))

(defn scale*
  ([type s] (apply-transform* type geom/scale s))
  ([type x y] (apply-transform* type geom/scale (geom/vec2d x y)))
  ([type x y z] (apply-transform* type geom/scale (geom/vec3d x y z))))

(defn scale-n*
  [type s] (apply-transform* type geom/scale-n s))

(defn translate*
  ([type t] (apply-transform* type geom/add t))
  ([type x y] (apply-transform* type geom/add (geom/vec2d x y)))
  ([type x y z] (apply-transform* type geom/add (geom/vec3d x y z))))

(defn rotate2d*
  [type theta] (apply-transform* type geom/rotate theta))

(defn center*
  ([type] (translate* type (geom/invert (geom/centroid type))))
  ([type origin] (translate* type (geom/sub origin (geom/centroid type)))))

(defn flip*
  [type] (assoc type :vertices (reverse (:vertices type))))

(defn transform2d-keys*
  [type keys matrix]
  (coll/apply-to-keys
    type keys
    (fn[_] (->> (geom/->3dxy _) (math/transform-point matrix) geom/->2dxy))
    nil))

(defn scale-keys*
  ([type keys s] (coll/apply-to-keys type keys geom/scale s))
  ([type keys x y] (scale-keys* type keys (geom/vec2d x y)))
  ([type keys x y z] (scale-keys* type keys (geom/vec3d x y z))))

(defn scale-n-keys*
  [type keys s] (coll/apply-to-keys type keys geom/scale-n s))

(defn translate-keys*
  ([type keys t] (coll/apply-to-keys type keys geom/add t))
  ([type keys x y] (translate-keys* type keys (geom/vec2d x y)))
  ([type keys x y z] (translate-keys* type keys (geom/vec3d x y z))))

(defn rotate2d-keys*
  [type keys theta] (coll/apply-to-keys type keys geom/rotate theta))

(defn center-keys*
  ([type keys] (translate-keys* type keys (geom/invert (geom/centroid type))))
  ([type keys origin] (translate-keys* type keys (geom/sub origin (geom/centroid type)))))

(defn line-flip
  [{:keys[a b]}] (assoc type :a b :b a))

(defn has-endpoint?
  ([coll p] (or (= (first coll) p) (= (last coll) p)))
  ([a b p] (or (= a p) (= b p))))

(defn point-on-line
  ([line ^double t] (geom/interpolate (:a line) (:b line) t))
  ([a b ^double t] (geom/interpolate a b t)))

(defn closest-point-to-line
  [{:keys[a b]} p]
  (let [v (geom/sub b a)
        t (/ (geom/dot (geom/sub p a) v) (geom/mag-squared v))]
    (cond
      (neg? t) a
      (> t 1.0) b
      :else (geom/interpolate a b t))))

(defn closest-point*
  [vertices p]
  (first (reduce
    (fn[[closest mind] e]
      (let[q (closest-point-to-line e p)
           d (geom/distance-squared q p)]
        (if (< d mind) [q d] [closest mind])))
    [nil Double/MAX_VALUE]
    (edges* vertices))))
  
(defn direction
  ([line] (geom/normalize (geom/sub (:b line) (:a line))))
  ([a b] (geom/normalize (geom/sub b a))))

(defn midpoint
  ([line] (geom/interpolate (:a line) (:b line) 0.5))
  ([a b] (geom/interpolate a b 0.5)))

(defn length
  [line] (geom/distance (:a line) (:b line)))

(defn length-squared
  [line] (geom/distance-squared (:a line) (:b line)))

(defn scale-length
  [line scale]
  (let [delta (* (- 1 scale) 0.5)
        a (:a line)
        b (:b line)]
    (assoc line :a (geom/interpolate a b delta) :b (geom/interpolate b a delta))))

(defn split-line
  ([line ^double len] (split-line line len [] true))
  ([line ^double len coll] (split-line line len coll true))
  ([line len coll addfirst?]
    (let [a (:a line) b (:b line)
          coll (if addfirst? (conj coll a) coll)
          dist (geom/distance a b)
          step (geom/limit (geom/sub b a) len)]
      (if (> dist len)
        (loop [pos (geom/add a step)
               acc coll
               d dist]
          (if (<= d len)
            (conj acc b)
            (recur (geom/add pos step) (conj acc pos) (- d len))))
        (conj coll b)))))

(defn extrude-vertices
  ([vertices dir amp] (extrude-vertices vertices :xyz dir amp))
  ([vertices plane dir amp]
  (let[amp (if-not (fn? amp) (constantly amp) amp)]
    (reduce
      (fn[mesh [p q i]]
        (let[a p
             b (geom/add p (geom/scale-n dir (amp i)))
             c (geom/add q (geom/scale-n dir (amp (inc i))))
             d q]
          (mesh/conjf mesh [[a c b] [a d c]])))
      (mesh/triangle-mesh3d)
      (map (fn[p q i] [(utils/swizzle p plane) (utils/swizzle q plane) i])
           (butlast vertices) (drop 1 vertices) (range))))))
