(ns toxi.geom.quadtree
  (:require
    [toxi.geom.core :as geom]
    [toxi.geom.rect :as rect]
    [clojure.zip :as zip]))

(defn node
  ([bounds] {:bounds bounds :children nil :type :empty})
  ([n children] (assoc n :children children :type :branch)))

(defn branch? [n] (not= :leaf (:type n)))

(defn children [n] (seq (:children n)))

(defn split
  [n]
  (let[{:keys[x y width height]} (:bounds n)
       w2 (* width 0.5)
       h2 (* height 0.5)
       cx (+ x w2)
       cy (+ y h2)]
    (assoc n :type :branch
           :children [(node (geom/rect x y w2 h2))
     (node (geom/rect cx y w2 h2))
     (node (geom/rect x cy w2 h2))
     (node (geom/rect cx cy w2 h2))])))

(defn qtree
  [bounds] (zip/zipper branch? children node (node bounds)))

(defn quadrant
  [loc p]
  (loop[c (zip/down loc)]
    (if-let [n (zip/node c)]
      (if (geom/contains-point? (:bounds n) p)
        c
        (recur (zip/right c))))))

(defn addpoint*
  [loc p]
  (let[{:keys[bounds type] :as n} (zip/node loc)]
    (prn bounds)
    (when (geom/contains-point? bounds p)
      (cond
        (= :empty type)
        (do (prn "empty") (zip/root (zip/replace loc (assoc n :point p :type :leaf))))
        (= :leaf type)
        (do (prn "leaf") 
        (if (= (:point n) p)
          (zip/root loc)
          (addpoint* (quadrant (zip/replace loc split) p) p)))
        (= :branch type)
        (do (prn "branch") (addpoint* (quadrant (zip/edit loc split) p) p))))))
        
(defn addpoint
  [n p]
  (addpoint* (zip/zipper branch? children node n) p))
