(ns open-scad.core
  (:refer-clojure :exclude [use import])
  (:require [clojure.java.io :refer [file]]
            [clojure.string :as str]
            [scad-clj.scad :as s]
            [scad-clj.model :as m]
            [shuriken.namespace :refer [import-namespace]]
            [shuriken.sequential :refer [update-nth]]
            [weaving.core :refer :all]
            [threading.core :refer :all]
            [open-scad.geometry]
            [com.palletops.ns-reload :refer [required-namespaces]]
            [dance.core :refer [dance]]))

(import-namespace scad-clj.scad)
(import-namespace scad-clj.model)
(import-namespace open-scad.geometry)

(def output-path "output")
(def libs-path   "libs")

(defonce inclusions (atom {}))

(defn ns-inclusions [ns]
  (let [ns     (the-ns  ns)
        ns-nme (ns-name ns)]
    (->> (cons ns-nme (required-namespaces ns-nme))
         (mapcat @inclusions)
         distinct)))

(defn definclusion [& incl]
  (swap! inclusions update (ns-name *ns*) (->| concat distinct) (flatten incl)))

;; TODO: move to shuriken
(defn update-last [s f]
  (update-nth s (dec (count s)) f))


(def ^:private separator ".")

(defgeometry part [name & block]
  (vec block))

(defn- part? [x]
  (and (list? x) (#{::part} (first x))))

(defn- part-name [part]
  (apply str (interpose separator (map name part))))

(defn- add-inclusions [& block]
  (vec (concat (map (fn [sym]
                      (let [pth (->> sym str (file libs-path)
                                     .getAbsolutePath)]
                        (if (some-> sym meta :use)
                          (m/use pth)
                          (m/include pth))))
                    (ns-inclusions (ns-name *ns*)))
               block)))

(defn- write+spit [part & block]
  (let [f (doto
            (apply file output-path
                   (-> *ns* str (str/split #"\.")
                       (update-last
                         (| str
                            (when (seq part) separator)
                            (apply str (part-name part))
                            ".scad"))))
            (-> .getParentFile .mkdirs))]
    (newline)
    (if (seq part)
      (println "--- Writing part" (part-name part) \newline
               "    in" (.getAbsolutePath f))
      (println "-- Rendering" (ns-name *ns*) \newline
               "   in" (.getAbsolutePath f)))
    (spit f (s/write-scad (apply add-inclusions block)))))

(def part-tracking-dance
  {:before (fn [form ctx]
             [form (if (part? form)
                     (update ctx :part conj (-> form second :name))
                     ctx)])
   :after  (fn [form ctx]
             [form (if (part? form)
                     (update ctx :part pop)
                     ctx)])})

(defn- $f-variable-assignment? [x]
  (and (list? x) (#{:open-scad.geometry/$fn
                    :open-scad.geometry/$fa
                    :open-scad.geometry/$fs}
                    (first x))))

(def $f-variables-tracking-dance
  {:scoped [:$fn :$fa :$fs]
   :before (fn [form ctx]
             [form (if ($f-variable-assignment? form)
                     (assoc ctx (-> form first name keyword) (-> form second :v))
                     ctx)])})

(defn render [& block]
  (write+spit nil block)
  ;; TODO
  ;; For now this works semi-ok, but in the end we need to separate:
  ;; - geometry construction
  ;; - geometry expansion
  ;; - geometry translation to scad format.
  (dance block
         part-tracking-dance
         $f-variables-tracking-dance
         :post (fn [form ctx]
                 (when (part? form)
                   (write+spit (:part ctx)
                               [(when-let [fn (:$fn ctx)] (fn! fn))
                                (when-let [fa (:$fa ctx)] (fa! fa))
                                (when-let [fs (:$fs ctx)] (fs! fs))
                                form]))
                 [form ctx]))
  )
