(ns open-scad.models.stackable-reel
  (:require [open-scad.core :refer :all]
            [open-scad.libs.parametric-star :refer [parametric-star]]
            [open-scad.libs.threads :refer [metric-thread]]
            [threading.core :refer :all]))

(def reel-diameter    25)
(def reel-height      (* 2 reel-diameter))
(def reel-thickness   2)
(def guard-diameter   (* 12/5 reel-diameter))
(def guard-height     3)
(def guard-holes-n    30)
(def screw-pitch      (/ guard-height 2))
(def screw-offset     0.8)
(def hook-diameter    8)
(def hook-height      guard-height)
(def hook-thickness   2)
(def hook-opening     1/4)
(def hook-bar-length  (- 8 hook-thickness))
(def pretty           false)


(defgeometry reel-hole []
  (->> (cylinder (- (/ (- reel-diameter screw-offset) 2) reel-thickness)
                 (* 2 reel-height)
                 :center false)
       (translate [0 0 (* -1/2 reel-height)])))

(defgeometry thread [& {:keys [internal]}]
  (let [d (- reel-diameter (/ reel-thickness 2))
        h (-> guard-height (when-> (<- internal) +ep +ep))]
    (-> (metric-thread (- d (if internal 0 screw-offset))
                       screw-pitch h
                       :internal internal
                       :leadin   (if internal 0 3))
        (when->> (<<- internal) (translate [0 0 ep])))))

(defgeometry reel []
  (let [h (- reel-height guard-height)]
    (difference
      (union (difference (cylinder (/ reel-diameter 2) h :center false)
                         (thread :internal true))
             (translate [0 0 h] (thread)))
      (reel-hole))))

(defn rand-pick [coll]
  (nth coll (Math/floor (rand (count coll)))))

(defgeometry guard-holes []
  (let [r         (/ guard-diameter 2)
        v         (range (- r) (+ r) 0.1)
        a         (range 0 360 0.1)
        sc        (range 0.5 1.5 0.1)
        positions (repeatedly guard-holes-n #(vector (rand-pick v)
                                                     (rand-pick v)
                                                     (rand-pick a)
                                                     (rand-pick sc)))]
    (apply union (for [[x y a sc] positions]
                   (->> (parametric-star 5 (* 2 guard-height) 1 2)
                        (rotate [0 0 (° a)])
                        (scale [sc sc 1])
                        (translate [x y (* -1/2 guard-height)]))))))

(defgeometry guard []
  (let [g (cylinder (/ guard-diameter 2) guard-height :center false)]
    (-> g
        (when-> (<- pretty) (difference (guard-holes)))
        (union (intersection g
                             (resize [(+ reel-diameter reel-thickness)
                                      (+ reel-diameter reel-thickness)
                                      0]
                                     (reel-hole))))
        (union (difference g
                           (resize [(- guard-diameter reel-thickness)
                                    (- guard-diameter reel-thickness)
                                    0]
                                   (reel-hole))))
        (difference (reel-hole)
                    (thread :internal true)))))

(defgeometry hook []
  (difference
    (cylinder (/ hook-diameter 2) hook-height :center false)
    (->> (cylinder (- (/ hook-diameter 2) hook-thickness)
                   (-> hook-height +ep +ep)
                   :center false)
         (translate [0 0 (- ep)]))
    (->> (square (+ep (/ hook-diameter 2))
                 (-> hook-height +ep +ep)
                 :center false)
         (extrude-rotate {:angle (* hook-opening 360)})
         (translate [0 0 (- ep)]))))

(defgeometry test-thread []
  (union
    (thread)
    (translate [-1.5 -5 0]
               (cube 3 10 6 :center false))
    (translate [(+ 5 reel-diameter) 0 0]
               (-> (reel)
                   (union (let [t reel-thickness
                                d (+ (/ reel-diameter 2) t)]
                            (difference (cylinder d reel-height :center false)
                                        (cylinder (- d t) (* 2 reel-height)))))
                   (difference (->> (cylinder reel-diameter (* reel-height 2)
                                              :center false)
                                    (translate [0 0 (* 2 guard-height)])))))))

(let [hk (->> (hook)
              (translate [0
                          (- (* -1/2 guard-diameter)
                             (* 1/2 hook-diameter)
                             (- hook-thickness))
                          0]))]
  (render ($fn 100 (union
                     (part :start-reel
                           (union (guard) (reel) hk))
                     (translate [(+ 5 guard-diameter) 0 0]
                                (part :reel (redef [pretty false]
                                                   (union (guard) (reel)))))
                     (translate [(* 2 (+ 5 guard-diameter)) 0 0]
                                (part :end-guard (union (guard) hk)))
                     #_(translate [(* 3 (+ 5 guard-diameter)) 0 0]
                                (part :test-thread (test-thread)))))))
