(ns open-scad.models.jar-bottom
  (:use clojure.pprint shuriken.macro)
  (:refer-clojure :exclude [use import])
  (:require [threading.core :refer :all]
            [weaving.core :refer :all]
            [open-scad.core :refer :all]
            [open-scad.libs.dot-scad :refer [polyline2d archimedean-spiral]]))

(def base-diameter 105)
(def base-height  3)
(def base-stripes 10)

(defn concentric-circles [d h n spacing & {:keys [invert]}]
  (let [r (/ d 2)]
    (apply union
           (for [x (range 0 r (/ r n))
                 :let [rx (- r x)]]
             (difference (cylinder rx h)
                         (scale +zep (cylinder (- rx spacing) h)))))))

(defn base [d h n & {:keys [cross-width spacing center-r]
                     :or {cross-width 5
                          spacing     (/ d n 4)
                          center-r    (+ (/ d n) spacing)}}]
  (let [r             (tap-> (/ d 2) (/ n) (> spacing) assert)
        crsp          (tap->> (+ cross-width spacing) (> center-r) assert)
        base-circle   (cylinder r        h)
        center-circle (cylinder (+ spacing center-r) h)
        core-hole     (cylinder center-r h)
        cross         (•- (intersection base-circle (cube d cross-width h))
                          (-> (union (rotate [0 0 (° 90)] (-•)))
                              (difference (scale +zep center-circle))))
        base          (concentric-circles d h n spacing)
        anti-base     (concentric-circles d h (/ n 2) (* 2 spacing))
        seg           (cube d spacing h)

        segments      (-> (union (translate [0 (* 1/2 crsp) 0] seg)
                                 (translate [0 (* -1/2 crsp) 0] seg))
                          (intersection base-circle))]
    (-> (union base
               (difference segments (scale +zep anti-base))
               (rotate [0 0 (° 90)] (intersection segments anti-base)))
        (difference  (union (scale +xyzep cross) (scale +zep center-circle)))
        (union (difference center-circle (scale +zep core-hole))))))

(defn holder-clip [r circumference spacing angle height length thick sphere-d]
  (let [sphere-r      (/ sphere-d 2)
        sphere-angle  (* (/ sphere-d circumference) 360)
        clip-spacing  (+ spacing sphere-r)
        clip-sphere   (->> (difference
                             (sphere sphere-r)
                             (translate [0 0 sphere-r]
                                        (cube sphere-d sphere-d sphere-d)))
                           (translate [(- r clip-spacing thick)
                                       0
                                       (- (/ height 2) thick)])
                           (rotate [0 0 (° (/ angle 2))]))
        extrude       #(->> %1
                            (translate [(- r (/ spacing 2)) 0])
                            (extrude-rotate {:angle     (* %2 angle)
                                             :convexity 10}))]
    (->> (difference
           (extrude (->> (square length height)
                         (translate [(/ thick -2) 0]))
                    1)
           (->> (extrude (->> (square (* 2 (+ spacing sphere-d)) base-height)
                              (translate [(* -2 thick (/ clip-spacing 2)) 0]))
                         2)
                (rotate [0 0 (° (/ angle -2))])))
         (union
           (rotate [0 0 (° (/ (- angle (* 5/4 sphere-angle)) 2))]
                   clip-sphere)
           (rotate [0 0 (° (/ (- 0 (- angle (* 5/4 sphere-angle))) 2))]
                   clip-sphere))
         (rotate [0 0 (° (* -1/2 angle))])
         #_(translate [(- (- r (/ spacing 2))) 0 0])
         #_(rotate [(° 180) (° -90) (° 90)]))))

(defn hygrometer-holder [d w thick]
  (let [w             20
        R             (/ base-diameter 2)
        r             (/ d 2)
        spacing       (/ R base-stripes 2)
        stand-width   5
        stand-height  2
        clip-thick    1
        circumference (* 2 π (+ R (* 2 spacing) clip-thick))
        clip-angle    (* (/ w circumference) 360)
        clip-sphere-d spacing
        clip-height   (+ #_0.4 (* 2 clip-thick) base-height)
        clip-length   (+ spacing clip-sphere-d clip-thick 4)
        clip          (holder-clip
                        R circumference spacing clip-angle clip-height
                        clip-length clip-thick clip-sphere-d)

        ; support       (-> (union (cylinder (+ r thick) w)
        ;                          (extrude-linear
        ;                            {:height w :convexity 20}
        ;                            (archimedean-spiral {:arm-distance   1.5
        ;                                                 :init-angle     360
        ;                                                 :point-distance 2
        ;                                                 :num-of-points  509
        ;                                                 :width          1})))
        ;                   (difference (->> (cylinder r (* 2 w))
        ;                                    (translate [0 0 -2])
        ;                                    (scale +zep))))
        ; stand         (cube stand-width stand-height w)
        ; pos           (- 0 r (/ stand-height 2) thick 3.1)
        ]
    ; (union support
    ;        ; (translate [0 pos 0] stand)
    ;        (rotate [0 (° 180) 0]
    ;                (translate [0
    ;                            (- pos (* 1/2 (-ep clip-height)))
    ;                            (- (/ clip-height 2) (+ep (- 1.4 (/ 0.4 2))))]
    ;                           clip)))
    clip))

(let [d base-diameter
      r (/ d 2)]
  (render ($fn 200 (union #_(base d base-height base-stripes)
                          (translate [base-diameter 0 0]
                                     (hygrometer-holder (+ 0.4 32.60) 5.5 2))))))
