(ns open-scad.models.peristaltic-pump
  (:require [open-scad.core :refer :all]
            [open-scad.libs.mcad.stepper :refer :all]
            [threading.core :refer :all]))

(def X
  {:bearings     {:d 7  :h 3 :hole-d 3 :count 6 :contact-count 3 :saliency 4}
   :nema         {:w 42.2 :l 42.2 :h 8 :round-extrusion {:d 22}}
   :latch        {:h 8 :basis {:h 2} :tube-offset 1}
   :wheel        {:r 15 :lid {:h 1.5} :center-cyl {:d 10 :screw {:d 1.5}}
                  :bearing-lid {:min-edge-dist 1 :h 1}}
   :tube         {:d 6 :inset-factor 1.05 :thickness 1}
   :nut          {:h 2.4 :D 6.2 :d 5.5 :dist-factor 0.9}
   :rail         {:w 5 :h 2 :l 30 :male-l 21.1
                  :groove {:w 2 :h 3 :l 30}
                  :margin 0.3
                  :dist-factor 1/3}
   :central-rail {:w 3.3 :h 5.5 :l 30 :margin 0.3 :pusher {:l 5}}
   :basis        {:h 7}})

(defn arc-chord-length [r a]
  (* 2 r (Math/sin (/ a 2))))

(defn arc-sagitta-length [r a]
  (- r (√ (- (** r) (** (* 1/2 (arc-chord-length r a)))))))

(defgeometry tube-groove []
  (let [r (-> X :wheel :r (* (-> X :tube :inset-factor))
              (- (+ #_(-> X :tube :d)
                    #_(-> X :tube :thickness (* 2))
                    #_(-> X :latch :tube-offset))))
        straight (->> (cylinder (-> X :tube :d (/ 2)) (-> X :nema :w))
                      (rotate [0 (° 90) 0])
                      (translate
                        [(-> X :nema :w (/ 2)) 0 0]))]
    (union (->> straight (translate [0 r 0]))
           (->> straight (translate [0 (- r) 0]))
           (->> (extrude-rotate {:angle 180}
                             (->> (circle (-> X :tube :d (/ 2)))
                                  (translate [r 0 0])))
             (rotate [0 0 (° 90)])))))

(defgeometry rail [& {:keys [l] :or {l (-> X :rail :l)}}]
  (->> (cube (-> X :rail :w) l (-> X :rail :h))
       (union (->> (cube (-> X :rail :groove :w)
                         l
                         (-> X :rail :groove :h))
                   (translate [0 0
                               (- (-> X :rail :groove :h
                                      (/ 2)))])))
       (rotate [(° 180) 0 (° 90)])
       (translate [(- (-> X :nema :l (- l) (/ 2)))
                   0
                   (- (+ (-> X :rail :h)
                         (-> X :rail :groove :h (/ 2)))
                      (-> X :latch :h (/ 2))
                      (-> X :rail :groove :h))])))

(defgeometry latch []
  (let [c          (-> X :nema (>-args (-> (cube :w :l (<- (-> X :latch :h))))))
        angle      (* (/ (* 2 π) (-> X :bearings :count))
                      (dec (-> X :bearings :contact-count)))
        sagitta    (arc-sagitta-length (-> X :wheel :r) angle)
        wheel-hole (cylinder (-> X :wheel :r) (-> X :latch :h (* 2)))
        offset     (+ sagitta (- (-> X :nema :w (/ 2))
                                 (-> X :wheel :r)))
        ; dig        (->> (cube 20 20 (-> X :latch :h (* 2)))
        ;                 (rotate [0 0 (° 45)])
        ;                 (translate [(- (+ 10 (-> X :nema :w (/ 3))))
        ;                             (- (+ 10 (-> X :nema :w (/ 3))))
        ;                             1]))
        ]
    (-> c
        (difference (->> c
                         (translate [offset 0 0])
                         (scale [1 1.1 1.1]))
                    wheel-hole)
        (difference (tube-groove))
        (->> (translate [0 0 (-> X :latch :h (/ 2) (+ 1))]))
        (union (->> (rail :l (-> X :rail :male-l))
                    (translate
                      [0 (-> X :nema :l (* (-> X :rail :dist-factor))) 0])
                    (>>- (-> (as-> $ (union $ (mirror [0 1 0] $)))))))

(union (-> (cube (-> X :nema :w)
                 (-> X :nema :l)
                 (-> X :latch :basis :h))
           (difference (->> (cube (-> X :nema :w)
                                  (-> X :nema :l (+ 1))
                                  (-> X :latch :basis :h (+ 1)))
                            (translate [(-> X :nema :w (/ 2)) 0 0]))))
       (-> (cube (-> X :central-rail :pusher :l)
                 (-> X :central-rail :w)
                 (-> X :central-rail :h)
                 :center false)
           (->> (translate
                  [(- (-> X :nema :l (/ 2)
                          (* (-> X :nut :dist-factor))
                          (- (-> X :nut :h (+ 0.01)))))
                   (- (-> X :central-rail :w (/ 2)))
                   (- (-> X :central-rail :h))]))))
        #_(difference dig
                      (mirror [0 1 0] dig)))))

(defgeometry basis []
  (let [$         (-> X :rail :margin)
        w         (+ $ (-> X :rail :l))
        l         (+ $ (-> X :rail :w))
        h         (+ $ (-> X :rail :groove :h))
        r         (-> (rail)
                      (minkowski (as-> (-> X :rail :margin) $
                                   (cube $ $ $)))
                      (->>
                        (translate
                          [0 (-> X :nema :l (* (-> X :rail :dist-factor))) 0])))
        central-r (-> X :central-rail (>-args (-> (cube :l :w :h)))
                      (->> (translate [(- (-> X :central-rail :l (/ 2)))
                                       0
                                       (- (-> X :central-rail :h (/ 2)))]))
                      (minkowski (as-> (-> X :central-rail :margin) $
                                   (cube $ $ $))))]
    (->> (cube (-> X :nema :w) (-> X :nema :l) (-> X :basis :h))
         (translate [0 0 (- (-> X :basis :h (/ 2)))])
         (>>- (-> (difference r
                              central-r
                              (mirror [0 1 0] r))
                  (difference (->> (cube (-> X :nut :h) (-> X :nut :d)
                                         (-> X :basis :h)
                                         :center false)
                                   (translate [(- (-> X :nema :w (/ 2)
                                                      (* (-> X :nut :dist-factor))))
                                               (- (-> X :nut :d (/ 2)))
                                               (- (-> X :central-rail :h))]))))))))

(defgeometry wheel [& {:keys [bearings]}]
  (let [r            (- (-> X :wheel :r) (-> X :bearings :saliency))
        lid-h        (-> X :wheel :lid :h)
        bear-lid-h   (-> X :wheel :bearing-lid :h)
        lid          (cylinder r lid-h)
        a            (/ 360 (-> X :bearings :count))
        hole         (cylinder (-> X :bearings :hole-d (/ 2))
                               (+ (-> X :bearings :h) (* 2 lid-h) 1))
        bearing      (-> (cylinder (-> X :bearings :d (/ 2))
                                   (-> X :bearings :h)
                                   :center false)
                         (difference (cylinder (-> X :bearings :hole-d (/ 2))
                                               (-> X :bearings :h (+ 1))
                                               :center false)))
        bearing-lid  (cylinder (-> X :bearings :hole-d (/ 2)
                                   (+ (-> X :wheel :bearing-lid :min-edge-dist)))
                               (+ lid-h bear-lid-h))
        offset       (-> X :bearings :d (/ 2))
        place        #(for [i (range (-> X :bearings :count))]
                        (->> %
                             (translate [(- (-> X :wheel :r) offset) 0 0])
                             (rotate [0 0 (° (* i a))])))
        holes        (place hole)
        bearings     (when bearings (place bearing))
        bearing-lids (place bearing-lid)
        drill-lid    (difference lid holes)
        bear-h       (-> X :bearings           :h)
        bearlid-h    (-> X :wheel :bearing-lid :h)
        lid-h        (-> X :wheel :lid         :h)
        center-cyl   (cylinder (-> X :wheel :center-cyl :d (/ 2))
                               (-> X :bearings :h)
                               :center false)
        cyl-screw    (->> (cylinder (-> X :wheel :center-cyl :screw :d (/ 2)) 7)
                          (rotate [0 (° 90) 0]))]
    (-> (union
          (->> drill-lid    (translate [0 0 (/ (+ bear-h bearlid-h lid-h)
                                                 2)]))
          (->> bearing-lids (translate [0 0 0]))
          center-cyl (when bearings bearings)
          (->> bearing-lids (translate [0 0 (/ (- 0 bear-h bearlid-h)
                                                 2)]))
          (->> drill-lid    (translate [0 0 (/ (- 0 bear-h bearlid-h lid-h)
                                                 2)])))
        (difference (->> (motor :model (literal "Nema17"))
                           (translate [0 0 5])
                           (union cyl-screw)
                           (rotate [0 0 (° (/ a 2))])))
        (->> (translate [0 0 (+ (-> X :latch :h (/ 2))
                                  (-> X :latch :basis :h (/ 2)))])))))

(defgeometry digs []
  (let [center-cyl (cylinder (-> X :nema :round-extrusion :d (/ 2))
                             (+ (-> X :basis :h)
                                (-> X :latch  :h)
                                10))]
    center-cyl))

(defgeometry hide-half []
  (->> (cube 40 40 20)
       (translate [0 0 (+ (-> X :latch :basis :h (/ 2))
                          (-> X :latch :h (/ 2))
                          10)])))

(render ($fn 40 (-> (union (latch) (basis))
                    (difference (digs))
                    (union (wheel :bearings false))
                    (difference (hide-half)))))
