(ns open-scad.geometry
  (:use clojure.pprint)
  (:require [clojure.walk :refer [walk postwalk]]
            [clojure.set :refer [difference]]
            [clojure.string :as str]
            [clojure.spec.alpha :as sp]
            [scad-clj.scad :as s]
            [scad-clj.model :as m]
            [shuriken.associative :refer [map-vals]]
            [shuriken.destructure :refer :all]
            [shuriken.macro :refer [is-form?]]
            [shuriken.sequential :refer [update-nth compact]]
            [shuriken.spec :refer [conform! either conf]]
            [spec-tools.visitor :refer [visit]]
            [spec-tools.parse :refer [parse-spec]]
            [threading.core :refer :all]
            [weaving.core :refer :all]
            [dance.core :refer [dance defdance parent-dance]]
            [lexikon.core :refer [lexical-eval]]))

(def ^:private ^:dynamic *semicolon* true)

(def  π     Math/PI)
(defn √     [x] (Math/sqrt x))
(defn **    [x] (* x x))
(defn °     [x] (* x 1/180 π))
(defn abs   [n] (max n (-' n)))
(def ep     1/100)
(defn +ep   [x] (+ x (* ep (abs x))))
(defn -ep   [x] (- x (* ep (abs x))))
(def +xep   [(+ep 1) 1 1])
(def +yep   [1 (+ep 1) 1])
(def +zep   [1 1 (+ep 1)])
(def -xep   [(-ep 1) 1 1])
(def -yep   [1 (-ep 1) 1])
(def -zep   [1 1 (-ep 1)])
(def +xyzep [(+ep 1) (+ep 1) (+ep 1)])
(def -xyzep [(-ep 1) (-ep 1) (-ep 1)])
(defn inch  [x] (* x 2.54))


(defn- seq-of? [pred x]
  (and (seq? x) (every? pred x)))

(defn clj->scad [form]
  (postwalk (fn [form]
              (when-> form ident?
                name
                (str/replace "-" "_")
                (if-> (<- (symbol? form)) symbol keyword)))
            form))

(def geometry?
  (and| seq?
        (->| count #{2})
        (->| first keyword?)))


(defmacro call [func args]
  ``(::call
      ~~(-> {:func (-> func clj->scad str)
             :args (-> args
                       (efface '&depth 'block)
                       (when-> (and-> (.contains '&) (-> last map?))
                         (->> (remove #{'&}) vec))
                       (restructure identity))}
            (when-> (<- (-> args deconstruct set (contains? 'block)))
              (assoc :block 'block)))))

(defmethod s/write-expr ::call [depth [_type {:keys [func args block]}]]
  (let [clean-args (->> (dance args
                               :pre   #(when-> % string? (as-> x (str \" x \")))
                               :walk? #(not (geometry? %))
                               :post  #(when->> % geometry?
                                         (s/write-expr depth)
                                         (binding [*semicolon* false])
                                         (apply str)))
                        (>>- (map-> (when->> map?
                                      (remove (->| second nil?))
                                      (into {}))))
                        (remove (and| map? empty?))
                        compact)]
    `[~(s/indent depth) ~func "(" ~(s/make-arguments (clj->scad clean-args)) ")"
      ~@(if block
          `[" {\n"
            ~@(s/write-block depth (when-not->> block (seq-of? seq?) list))
            ~(s/indent depth) "}\n"]
          (when *semicolon* [";\n"]))]))


(defmacro defgeometry* [nme & {c-args+bodies    :constructor
                                [w-args & w-body] :writer}]
  (let [c-args+bodies (when-not->> c-args+bodies (seq-of? seq?) list)]
    `(do (defn ~nme ~@c-args+bodies)
         (defmethod s/write-expr ~(keyword (str *ns*) (str nme))
           [~'&depth ~w-args]
           ~@w-body))))

(defn- apply->| [f & fns]
  (apply ->| f (map apply| fns)))

(defn- form| [f]
  (fn [form ctx]
    [(f form) ctx]))

(defn- ctx| [f]
  (fn [form ctx]
    [form (f ctx)]))

(defn form-ctx| [f]
  (fn [form ctx]))

(defdance spec-dance
  :before (when| sp/spec? (->| sp/form eval))
  :walk?  (and| sp/regex?))

(defmacro defgeometry
  ;; Case 1: bind a function defined in scad-clj.model
  ([nme]
   (if-let [args (some->> (str nme) (symbol "m") resolve meta :arglists last
                          (postwalk (when| (and| map? (| contains? :or))
                                      (| dissoc :or))))]
     `(defgeometry ~nme ~args)
     (throw (ex-info (str nme " does not seem to be a var of ns scad-clj.model")
                     {:type ::geometry-var-not-found
                      :sym nme}))))
  ;; Case 2: bind an existing OpenScad module
  ([nme args]
   (let [foreign-name (or (-> nme meta :foreign-name) nme)
         gen-expr (fn [args]
                    (or (some-> (symbol "m" (str nme)) resolve .toSymbol)
                        (when (vector? args)
                          `(call ~foreign-name
                                 ~(vec (concat '[&depth &opts] args))))
                        ;; assume it's a seq
                        `(let [spc# ~(lexical-eval args)
                               ks# (->> (conform! spc# args#)
                                        (postwalk (when| map-entry?
                                                    (juxt identity #(do `'~%))
                                                    vec))
                                        (sp/unform spc#))]
                           (println "------------> ks:" ks#))))]
     (if (seq-of? vector? args)
       `(defgeometry ~nme ~@(map  #(do `(~% ~(gen-expr %)))  args))
       `(defgeometry ~nme ~args ~(gen-expr args)))))
  ;; Case 3: define your own geometry
  ([nme more+ & +more]
   (let [args+body-spec (sp/cat :args vector? :body (sp/* any?))
         {:keys [params args+bodies]}
         (conform!
           (sp/cat
             :params    (sp/* (sp/cat :k keyword? :v any?))
             :args+bodies (conf (sp/alt
                                  :single   args+body-spec
                                  :multiple (sp/+ (sp/spec args+body-spec))
                                  :spec-arg (sp/cat :spec (or| ident? list?)
                                                    :body (sp/* any?)))
                                (|| conj {})
                                first))
           (cons more+ +more))
         defaults {:write-expr true}
         params (merge defaults (-> params
                                    (map-> (juxt-> :k :v) vec)
                                    (->> (into {}))))
         ; _ (println "-------------------------------")
         ; _ (pprint (-> args+bodies :spec-arg :spec))
         ; _ (println "-------------------------------")
         ; _ (pprint (-> args+bodies :spec-arg :spec eval))
         ; _ (println "-------------------------------")
         ; _ (pprint (dance spec-dance))
         ; _ (println "====================================")

         ; _ (pprint (-> (dance (-> args+bodies :spec-arg :spec eval)
         ;                      parent-dance
         ;                      :context {:ks []}
         ;                      :return :context
         ;                      :walk? (fn [form ctx]
         ;                               (if (map-entry? form)
         ;                                 [(and (-> ctx :parent (contains? :clojure.spec.alpha/op))
         ;                                       (#{:forms :ks} (key form)))
         ;                                  ctx]
         ;                                 [true ctx]))
         ;                      :pre (fn [form ctx]
         ;                             (if (and (map-entry? form)
         ;                                      (-> ctx :parent (contains? :clojure.spec.alpha/op))
         ;                                      (= :form (key form)))
         ;                               [(eval form) ctx]
         ;                               [form ctx]))
         ;                      )
         ;               :ks))
         args+bodies
         (-> args+bodies
             (when-> :single
               :single (juxt-> :args :body) (->> (apply cons)) list)
             (when-> :multiple
               :multiple (map-> (juxt-> :args :body) (->> (apply cons))))
             (when-> :spec-arg
               :spec-arg (•- (<- (do `(([& args#]
                                        (let [ks# (->> (conform! ~(-• :spec) args#)
                                                       #_vals
                                                       #_(apply merge)
                                                       #_(map (fn [[k# _v#]]
                                                              [k# k#]))
                                                       #_(into {}))]
                                          (println "ici:")
                                          (pprint ks#)))))))))
         [nme macro-nme] (if (:macro params)
                           [(symbol (str nme \*))  nme]
                           [nme])
         helper-name     (symbol (str "_write-" nme))]
     `(do (declare ~nme)
          (defn- ~helper-name
            ~@(map (fn [[args & body]]
                     `([~'&depth ~'&opts ~@args]
                       ~(if (:write-expr params)
                          `(s/write-expr ~'&depth (do ~@body))
                          `(do ~@body))))
                   args+bodies))
          (defgeometry* ~nme
            :constructor ~(for [[args & _body] args+bodies]
                            `(~args
                               (list ~(keyword (str *ns*) (str nme))
                                     (-> (->> ~(deconstruct args :as-map true)
                                              (remove (->| second nil?))
                                              (into {}))
                                         (assoc :argv '~args)))))
            :writer      ([_type# {:keys [~'argv] :as m#}]
                          (let [argsm#       (->> (dissoc m# :argv)
                                                  (remove (->| second nil?))
                                                  (into {}))
                                opts#        {:keys
                                              (-> (difference
                                                    (->> (keys m#)
                                                         (remove #{:argv})
                                                         (map symbol)
                                                         set)
                                                    (set (deconstruct ~'argv)))
                                                  vec)}
                                helper-args# (restructure
                                               (vec (concat ['~'&depth opts#]
                                                            ~'argv))
                                               (assoc argsm#
                                                 '~'&depth ~'&depth))]
                            (apply ~helper-name helper-args#))))
          ~@(when-let [m-args+bodies (some-> (:macro params)
                                             (when-not->> (seq-of? seq?) list))]
              [`(defmacro ~macro-nme ~@m-args+bodies)])))))


(defn literal [& vs]
  `(::literal {:vs ~(vec vs)}))

(defmethod s/write-expr ::literal [depth [_type {:keys [vs]}]]
  (->> vs
       (map (when| geometry? (|| s/write-expr depth)))
       (apply str)))

(defgeometry bind
  :write-expr false
  :macro ([bindings & block]
          `(binding ~bindings
             (bind* ~(->> (partition 2 bindings)
                          (mapcat (fn [[sym expr]]  `[(var ~sym) ~expr]))
                          vec)
               ~@block)))
  [bindings & block]
  (let []
    (push-thread-bindings (apply hash-map bindings))
    (try
      (s/write-block &depth block)
      (finally
        (pop-thread-bindings)))))

(defgeometry redef
  :write-expr false
  :macro ([bindings & block]
          `(with-redefs ~bindings
             (redef* ~(->> (partition 2 bindings)
                           (mapcat (fn [[sym expr]]  `[(var ~sym) ~expr]))
                           vec)
                     (fn [] ~(vec block)))))
  [bindings block-fn]
  (let [binding-map (apply hash-map bindings)
        root-bind (fn [m]
                    (doseq [[a-var a-val] m]
                      (.bindRoot ^clojure.lang.Var a-var a-val)))
        old-vals (zipmap (keys binding-map)
                         (map #(.getRawRoot ^clojure.lang.Var %) (keys binding-map)))]
    (try
      (root-bind binding-map)
      (s/write-block &depth (block-fn))
      (finally
        (root-bind old-vals)))))

(defgeometry assign
  :macro ([nme expr]  `(assign* '~nme (bind [*semicolon* false]
                                        ~(if (ident? expr) `'~expr expr))))
  [nme expr]
  (literal (str (-> nme  clj->scad name) " = "
                (-> expr
                    (when->> ident?    clj->scad name)
                    (when->> geometry? (s/write-expr &depth) (apply str)))
                (when *semicolon* ";\n"))))

(defgeometry $fn [v & block]  (cons (m/fn! v) block))
(defgeometry $fa [v & block]  (bind [m/*fa* v] (cons (m/fa! m/*fa*) block)))
(defgeometry $fs [v & block]  (bind [m/*fs* v] (cons (m/fs! m/*fs*) block)))

(defmulti width     first)
(defmulti length    first)
(defmulti height    first)
(defmulti radius    first)
(defmulti thickness first)
(defmulti opening   first)

(defmethod width     ::$fn [[_type {:keys [block]}]]  (-> block last width))
(defmethod length    ::$fn [[_type {:keys [block]}]]  (-> block last length))
(defmethod height    ::$fn [[_type {:keys [block]}]]  (-> block last height))
(defmethod thickness ::$fn [[_type {:keys [block]}]]  (-> block last thickness))

(defmethod width  :cube [[_type {:keys [x]}]]  x)
(defmethod length :cube [[_type {:keys [y]}]]  y)
(defmethod height :cube [[_type {:keys [z]}]]  z)

(defmethod width  :cylinder [[_type {:keys [r]}]]  (* r 2))
(defmethod length :cylinder [[_type {:keys [r]}]]  (* r 2))
(defmethod height :cylinder [[_type {:keys [h]}]]  h)
