(ns open-scad.libs.relativity
  "Help: https://github.com/davidson16807/relativity.scad/wiki"
  (:use clojure.pprint)
  (:require [open-scad.core :refer :all]
            [shuriken.sequential :refer [assoc-nth-in]]
            [threading.core :refer :all]))

(definclusion 'relativity.scad)

(defgeometry box  ([[x y z]] [[x y z] block]))
(defgeometry rod  ([r h]     [r h block]))
(defgeometry ball ([r]       [r block]))

(defgeometry align  [v block])
(defgeometry orient [v block])
(defgeometry parent [])

(defn        opts        [k v obj]   (assoc-nth-in obj [1 k]  v))
(defn        anchored    [v obj]     (opts :anchor      v     obj))
(defn        oriented    [v obj]     (opts :orientation v     obj))
(defn        named       [klass obj] (opts :$class      klass obj))
(defgeometry parent-size []          (literal "$parent_size"))

(def top    [0 0 1])
(def center [0 0 0])
(def bottom [0 0 -1])
(def x      [1 0 0])
(def y      [0 1 0])
(def z      [0 0 1])

(def inward  (literal "$inward"))
(def outward (literal "$outward"))

(defgeometry show         ([block]
                           [selector block]))
(defgeometry hide         ([block]
                           [selector block]))
(defgeometry differed     ([negative block]
                           [negative positive block]
                           [negative positive unaffected block]))
(defgeometry hulled       ([block]
                           [selector block]))
(defgeometry intersected  ([selector-a selector-b block]
                           [selector-a selector-b unaffected block]))
(defgeometry rotated      ([offset block]
                           [offset n block]
                           [offset n selector block]))
(defgeometry translated   ([offset block]
                           [offset n block]
                           [offset n selector block]))
(defgeometry scaled       ([v block]
                           [v selector block]))
(defgeometry resized      ([v block]
                           [v selector block]))
(defgeometry mirrored     ([v block]
                           [v selector block]))
(defgeometry colored      ([color block]
                           [color selector block]))

(defgeometry selectable ([block]
                         [{:keys [$class]} block]))

(defn not-$class [s]
  (str "not(" s ")"))


(defgeometry div* [[x y z] {:keys [anchor orientation shown name $class]} block]
  (let [div-name     (or name $class (str (gensym "div")))
        debug-name   (str div-name "-debug")
        content-name (str div-name "-content")]
    (-> (named div-name
               (box [x y z]
                    [(selectable {:$class content-name} block)
                     (when shown (-% (named debug-name (box [x y z]))))]))
        (when->> (<<- anchor)      (anchored      anchor))
        (when->> (<<- orientation) (oriented orientation))
        (->> (hide div-name)))))

(defn div
  ([dims block]       (div dims {} block))
  ([dims opts block]  (div* dims opts block)))
